<?php

namespace ILC\PermissionCenter\Controllers;

use Spatie\Permission\Models\Role AS Rol;
use Spatie\Permission\Models\Permission AS Permiso;
use App\Models\User AS Usuario;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use ILC\PermissionCenter\Services\UserLevel;

class UsuarioController implements HasMiddleware
{

    public static function middleware(): array
    {
        return [
            new Middleware('can:listar.roles', only:['index', 'show']),
            new Middleware('can:asignar.permisos', only:['darPermisosUsuario']),
            new Middleware('can:asignar.roles', only:['darRolesUsuario']),
            new Middleware('can:retirar.permisos', only:['quitarPermisoUsuario']),
            new Middleware('can:retirar.roles', only:['quitarRolUsuario']),
        ];
    }
	public function index(Request $request)
    {
		$usuarios = Usuario::with([
            'permissions' => fn($permission) => $permission->select(['id', 'name']),
			'roles' => fn($rol) => $rol->select(['id', 'name']),
		])->when(!empty($request->filtro), function($query) use ($request){
            $query->whereRelation('permisos', 'name', 'like', "%$request->filtro%")
            ->orWhereRelation('roles', 'name', 'like', "%$request->filtro%")
            ->orWhere('nombre', 'like', "%$request->filtro%");
        });

		return $usuarios->paginate($request->get('perPage', 10));
	}

	public function darPermisosUsuario(Request $request, Usuario $usuario)
	{
		$request->validate([
            'permisos' => ['required', 'array', 'user_can_be_modified', 'can_assign_permissions'],
			'permisos.*' => ['exists:permissions,id'],
		]);

        $permisos = Permiso::findMany($request->permisos);

		$data = $usuario->givePermissionTo($permisos);

		return new JsonResponse(['message' => 'permiso(s) asignado(s)', $data], 201);
	}

	public function darRolesUsuario(Request $request, Usuario $usuario)
	{
		$request->validate([
			'roles' => ['required', 'array', 'user_can_be_modified', 'can_assign_roles'],
			'roles.*' => ['exists:roles,id'],
		]);

        $roles = Rol::findMany($request->roles);

		$data = $usuario->assignRole($roles);

		return new JsonResponse(['message' => 'rol(es) guardado(s)', $data], 201);
	}

	public function show(Usuario $usuario)
	{
		$data =  $usuario->load([
			'permissions',
			'roles'
		]);

        return new JsonResponse(['data' => $data], 200);
	}

	public function quitarPermisoUsuario(Usuario $usuario, Permiso $permiso)
	{
        if (config('permission-center.niveles', true)) {
            $userLevel = UserLevel::getFor(auth()->user());
            if ($userLevel > UserLevel::getFor($usuario) || $userLevel > $permiso->level) {
                abort(403, 'No tienes la jerarquía suficiente para quitar este permiso de este usuario.');
            }
        }

		return $usuario->revokePermissionTo($permiso->name);
	}

	public function quitarRolUsuario(Usuario $usuario, Rol $rol)
	{
        if (config('permission-center.niveles', true)) {
            $userLevel = UserLevel::getFor(auth()->user());
            if ($userLevel > UserLevel::getFor($usuario) || $userLevel > $rol->level) {
                abort(403, 'No tienes la jerarquía suficiente para quitar este rol de este usuario.');
            }
        }

		return $usuario->removeRole($rol->id);
	}
}
