<?php

namespace ILC\PermissionCenter;

use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use ILC\PermissionCenter\Commands\PermissionCenterCommand;
use ILC\PermissionCenter\Rules\CanAssignPermissionsRule;
use ILC\PermissionCenter\Rules\CanAssignRolesRule;
use ILC\PermissionCenter\Rules\LevelRule;
use ILC\PermissionCenter\Rules\UserCanBeModifiedRule;

class PermissionCenterServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        $package
            ->name('permission-center')
            ->hasConfigFile()
            ->hasMigrations(['modify_permissions_tables', 'add_level_columns_to_permissions_tables'])
            ->hasCommand(PermissionCenterCommand::class);
    }

    protected function registerRoutes(): void
    {
        $this->loadRoutesFrom(__DIR__.'/routes/api.php');
    }

    public function boot()
    {
        parent::boot();

        $this->registerRoutes();
        $this->registerValidationRules();

        Gate::before(function ($user, $ability) {
            return $user->hasRole('Super usuario') ? true : null;
        });

        $this->overrideAdminUsuariosConfig();
    }

    protected function overrideAdminUsuariosConfig(): void
    {
        if (config('permission-center.niveles') && class_exists(\ILC\AdminUsuarios\AdminUsuariosServiceProvider::class)) {

            $configKey = 'adminusuarios.validation_rules';
            $existingRules = config($configKey, []);

            $newRules = [
                'create' => [
                    'roles' => [
                        'user_can_be_modified',
                        'can_assign_roles'
                    ],
                    'permisos' => [
                        'can_assign_permissions'
                    ]
                ],
                'update' => [
                    'roles' => [
                        'user_can_be_modified',
                        'can_assign_roles'
                    ],
                    'permisos' => [
                        'can_assign_permissions'
                    ]
                ],
            ];

            $mergedRules = array_merge_recursive($existingRules, $newRules);

            config([$configKey => $mergedRules]);
        }
    }

    protected function registerValidationRules(): void
    {
        Validator::extend('level_rule', function ($attribute, $value, $parameters, $validator) {
            return (new LevelRule())->passes($attribute, $value);
        });

        Validator::extend('user_can_be_modified', function ($attribute, $value, $parameters, $validator) {
            $rule = new UserCanBeModifiedRule();
            if (!$rule->passes($attribute, $value)) {
                $validator->errors()->add($attribute, $rule->message());
                return false;
            }
            return true;
        });

        Validator::extend('can_assign_roles', function ($attribute, $value, $parameters, $validator) {
            $rule = new CanAssignRolesRule();
            if (!$rule->passes($attribute, $value)) {
                $validator->errors()->add($attribute, $rule->message());
                return false;
            }
            return true;
        });

        Validator::extend('can_assign_permissions', function ($attribute, $value, $parameters, $validator) {
            $rule = new CanAssignPermissionsRule();
            if (!$rule->passes($attribute, $value)) {
                $validator->errors()->add($attribute, $rule->message());
                return false;
            }
            return true;
        });
    }
}
