<?php

namespace ILC\PermissionCenter\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Spatie\Permission\Models\Role as Rol;
use Spatie\Permission\Models\Permission as Permiso;
use ILC\PermissionCenter\Events\RolCreado;
use ILC\PermissionCenter\Events\RolActualizado;
use ILC\PermissionCenter\Events\RolEliminado;
use ILC\PermissionCenter\Events\RolPermisosSincronizados;
use ILC\PermissionCenter\Queries\RolIndexQuery;
use ILC\PermissionCenter\Services\UserLevel;

class RolController implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('can:listar.roles', only:['index', 'show']),
            new Middleware('can:crear.roles', only:['store']),
            new Middleware('can:editar.roles', only:['update']),
            new Middleware('can:borrar.roles', only:['destroy']),
            new Middleware('can:asignar.permisos', only:['sincronizarPermisos']),
        ];
    }
	public function index(Request $request, RolIndexQuery $query)
	{
        $data = $query($request);

        return $data->paginate($request->get('perPage', 10));
	}

	public function store(Request $request)
	{
        $validationRules = [
            'name' => ['required', 'regex:/^[a-zA-Z0-9\s\.\-]+$/u'],
            'module' => ['required'],
            'description' => ['required'],
        ];

        $dataToCreate = ['name', 'module', 'description'];

        if (config('permission-center.niveles')) {
            $validationRules['level'] = ['required', 'integer', 'level_rule'];
            $dataToCreate[] = 'level';
        }

        $request->validate($validationRules);

        $data = $request->only($dataToCreate);
        $data['guard_name'] = 'web';

		$rol = Rol::create($data);

        event(new RolCreado($rol));
        return $rol;
	}

	public function update(Request $request, Rol $rol)
	{
        $validationRules = [
            'name' => ['required', 'regex:/^[a-zA-Z0-9\s\.\-]+$/u'],
            'module' => ['required'],
            'description' => ['required'],
        ];

        $dataToUpdate = ['name', 'module', 'description'];

        if (config('permission-center.niveles')) {
            if (UserLevel::getFor(auth()->user()) > $rol->level) {
                abort(403, 'No tienes la jerarquía suficiente para editar este rol.');
            }
            $validationRules['level'] = ['required', 'integer', 'level_rule'];
            $dataToUpdate[] = 'level';
        }

        $request->validate($validationRules);

        $rol->update($request->only($dataToUpdate));
        event(new RolActualizado($rol));

        return new JsonResponse(['mensaje'=> 'Actualizado correctamente']);
	}

	public function destroy(Rol $rol)
	{
        if (config('permission-center.niveles')) {
            if (UserLevel::getFor(auth()->user()) > $rol->level) {
                abort(403, 'No tienes la jerarquía suficiente para eliminar este rol.');
            }
        }

        event(new RolEliminado($rol));
		$rol->delete();
        return new JsonResponse(['mensaje'=> 'Eliminado correctamente']);
	}

	public function sincronizarPermisos(Request $request, Rol $rol)
	{
		$request->validate([
			'permisos' => 'required|array',
			'permisos.*' => 'nullable|exists:permissions,id',
		]);

        $permisosIds = $request->permisos;
        $permisos = Permiso::findMany($permisosIds);

        if (config('permission-center.niveles')) {
            $userLevel = UserLevel::getFor(auth()->user());
            if ($userLevel > $rol->level) {
                abort(403, 'No tienes la jerarquía suficiente para modificar este rol.');
            }

            foreach ($permisos as $permiso) {
                if ($userLevel > $permiso->level) {
                    abort(403, "No tienes la jerarquía suficiente para asignar el permiso \"$permiso->name\".");
                }
            }
        }

		$rol->syncPermissions($permisos);
        event(new RolPermisosSincronizados($rol, $permisos->pluck('id')->toArray()));

        return new JsonResponse(['mensaje'=> 'Permisos sincronizados correctamente']);
	}
}
