<?php

namespace ILC\PermissionCenter\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Spatie\Permission\Models\Permission as Permiso;
use ILC\PermissionCenter\Events\PermisoCreado;
use ILC\PermissionCenter\Events\PermisoActualizado;
use ILC\PermissionCenter\Events\PermisoEliminado;
use ILC\PermissionCenter\Queries\PermisoIndexQuery;
use ILC\PermissionCenter\Services\UserLevel;

class PermisoController implements HasMiddleware
{

    public static function middleware(): array
    {
        return [
            new Middleware('can:listar.permisos', only:['index', 'show']),
            new Middleware('can:crear.permisos', only:['store']),
            new Middleware('can:editar.permisos', only:['update']),
            new Middleware('can:borrar.permisos', only:['destroy']),
        ];
    }

    public function index(Request $request, PermisoIndexQuery $query)
	{
        $data = $query($request);
		return $data->paginate($request->get('perPage', 10));
	}

	public function store(Request $request)
	{
        $validationRules = [
            'name' => ['required', 'regex:/^[a-zA-Z0-9\s\.\-]+$/u'],
            'module' => ['required'],
            'description' => ['required'],
        ];

        $dataToCreate = ['name', 'module', 'description'];

        if (config('permission-center.niveles')) {
            $validationRules['level'] = ['required', 'integer', 'level_rule'];
            $dataToCreate[] = 'level';
        }

        $request->validate($validationRules);

        $data = $request->only($dataToCreate);
        $data['guard_name'] = 'web';

		$permiso = Permiso::create($data);

        event(new PermisoCreado($permiso));
        return $permiso;
	}

	public function update(Request $request, Permiso $permiso)
	{
        $validationRules = [
            'name' => ['required', 'regex:/^[a-zA-Z0-9\s\.\-]+$/u'],
            'module' => ['required'],
            'description' => ['required'],
        ];

        $dataToUpdate = ['name', 'module', 'description'];

        if (config('permission-center.niveles')) {
            if (UserLevel::getFor(auth()->user()) > $permiso->level) {
                abort(403, 'No tienes la jerarquía suficiente para editar este permiso.');
            }
            $validationRules['level'] = ['required', 'integer', 'level_rule'];
            $dataToUpdate[] = 'level';
        }

        $request->validate($validationRules);

		$permiso->update($request->only($dataToUpdate));
        event(new PermisoActualizado($permiso));
        return new JsonResponse(['mensaje'=> 'Actualizado correctamente']);
	}

	public function destroy(Permiso $permiso)
	{
        if (config('permission-center.niveles')) {
            if (UserLevel::getFor(auth()->user()) > $permiso->level) {
                abort(403, 'No tienes la jerarquía suficiente para eliminar este permiso.');
            }
        }

        event(new PermisoEliminado($permiso));
		$permiso->delete();
        return new JsonResponse(['mensaje'=> 'Eliminado correctamente']);
	}
}
