<?php

namespace ILC\MenuCenter\Models;

use ILC\MenuCenter\Database\Factories\MenuFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use OwenIt\Auditing\Auditable as AuditableTrait;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Support\Facades\Cache;
use ILC\MenuCenter\Events\{
    MenuCreado, MenuModificado, MenuEliminado, MenuActivado, MenuDesactivado, MenuItemCreado,
    MenuItemModificado, MenuItemEliminado, MenuItemActivado, MenuItemDesactivado
};

class Menu extends Model implements Auditable
{
    use HasFactory;
    use AuditableTrait;


    protected $fillable = [
        'parent_id',
        'name',
        'route',
        'icon',
        'order',
        'description',
        'separator',
        'layout',
        'enabled',
        'permission_name',
        'icon_name',
        'menu_root_id'
    ];


    protected $dispatchesEvents = [
        'MenuItemCreado' => MenuItemCreado::class,
        'MenuItemModificado' => MenuItemModificado::class,
        'MenuItemEliminado' => MenuItemEliminado::class,
        'MenuItemActivado' => MenuItemActivado::class,
        'MenuItemDesactivado' => MenuItemDesactivado::class,

    ];
    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function menu_root(): BelongsTo
    {
        return $this->belongsTo(MenuRoot::class, 'menu_root_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($model) {
            $model->enabled = $model->enabled ?? true;
        });
    }

    protected static function booted()
    {
        static::saved(function ($menu) {
            $cacheKey = "menu_breadcrumbs_{$menu->id}";
            Cache::forget($cacheKey);
        });

        static::deleted(function ($menu) {
            $cacheKey = "menu_breadcrumbs_{$menu->id}";
            Cache::forget($cacheKey);
        });
    }

    public function getPath(): array
    {
        $path = [];
        $current = $this;

        while ($current) {
            $path[] = [
                'id' => $current->id,
                'name' => $current->name,
                'to' => [
                    'path' => $current->route ?? '/',
                ],
            ];
            $current = $current->parent;
        }

        return array_reverse($path);
    }

    protected static function newFactory()
    {
        return MenuFactory::new();
    }
}
