<?php

declare(strict_types=1);

namespace ILC\MenuCenter\Http\Controllers;

use ILC\MenuCenter\Events\MenuActivado;
use ILC\MenuCenter\Events\MenuCreado;
use ILC\MenuCenter\Events\MenuEliminado;
use ILC\MenuCenter\Events\MenuItemActivado;
use ILC\MenuCenter\Events\MenuItemCreado;
use ILC\MenuCenter\Events\MenuItemDesactivado;
use ILC\MenuCenter\Events\MenuItemEliminado;
use ILC\MenuCenter\Events\MenuItemModificado;
use ILC\MenuCenter\Events\MenuModificado;
use ILC\MenuCenter\Http\Requests\MenuRequest;
use ILC\MenuCenter\Http\Requests\MenuRootRequest;
use ILC\MenuCenter\Models\Menu;
use ILC\MenuCenter\Models\MenuRoot;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;

class MenuController implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware(['auth:sanctum','can:menu-center-listar-menus'], only: ['getMenuTree', 'getBreadcrumbs', 'getMenuRoot']),
            new Middleware(['auth:sanctum','can:menu-center-crear-menus'], only: ['createMenuRoot', 'createMenuItem']),
            new Middleware(['auth:sanctum','can:menu-center-editar-menus'], only: ['updateMenuRoot', 'updateMenuItem']),
            new Middleware(['auth:sanctum','can:menu-center-eliminar-menus'], only: ['deleteMenuRoot', 'deleteMenuItem'])
        ];
    }
    public function getMenuRoot($id)
    {
        $menuRoot = MenuRoot::findOrFail($id);

        return response()->json($menuRoot);
    }

    public function createMenuRoot(MenuRootRequest $request): JsonResponse
    {
        $menuRoot = MenuRoot::create($request->all());
        MenuCreado::dispatch($menuRoot);
        return response()->json($menuRoot, 201);
    }

    public function updateMenuRoot(MenuRootRequest $request, $id): JsonResponse
    {
        $menuRoot = MenuRoot::findOrFail($id);
        $wasEnabled = $menuRoot->enabled;

        $menuRoot->update($request->all());
        $menuRoot->save();

        if ($wasEnabled !== $menuRoot->enabled) {
            $this->updateChildrenEnabledStatus($menuRoot, $menuRoot->enabled);
            $menuRoot->enabled ? MenuActivado::dispatch($menuRoot) : MenuEliminado::dispatch($menuRoot);
        }

        MenuModificado::dispatch($menuRoot);
        return response()->json($menuRoot);
    }

    public function deleteMenuRoot($id)
    {
        $menuItem = MenuRoot::findOrFail($id);
        $menuItem->delete();
        MenuEliminado::dispatch($menuItem);

        return response()->noContent();
    }

    public function createMenuItem(MenuRequest $request)
    {
        $parentId = $request->get('parent_id', null);
        $last = Menu::where('parent_id', $parentId)->where('menu_root_id', $request->menu_root_id)->count() + 1;
        $menuItem = Menu::create(array_merge(['order' => $last], $request->all()));
        MenuItemCreado::dispatch($menuItem);
        return response()->json($menuItem, 201);
    }

    public function updateMenuItem(MenuRequest $request, $id)
    {
        $menuItem = Menu::findOrFail($id);
        $menuRootId = $menuItem->menu_root_id;
        $oldParentId = $menuItem->parent_id;
        $oldOrder = $menuItem->order;
        $wasEnabled = $menuItem->enabled;
        $data = $request->all();

        if ($oldOrder !== $request->order) {
            $oldMenu =  Menu::where('parent_id', $menuItem->parent_id)->where('menu_root_id', $menuRootId)->where('order', $request->order)->first();
            if (null !== $oldMenu) {
                $oldMenu->order = $oldOrder;
                $oldMenu->save();
            } else {
                $data['order'] = $oldOrder;
            }
        }

        if ($oldParentId !== $request->parent_id) {
            $menus = Menu::where('parent_id', $request->parent_id)->where('menu_root_id', $menuRootId)->orderBy('order', 'ASC')->get();
            foreach ($menus as $index => $menu) {
                $menu->order = $index + 1;
                $menu->save();
            }
            $data['order'] = count($menus) + 1;
        }

        $menuItem->update($data);
        $menuItem->save();
        if ($oldParentId !== $request->parent_id) {
            $menusOld = Menu::where('parent_id', $oldParentId)->where('menu_root_id', $menuRootId)->orderBy('order', 'ASC')->get();
            foreach ($menusOld as $index => $menu) {
                $menu->order = $index + 1;
                $menu->save();
            }
        }
        if ($wasEnabled !== $menuItem->enabled) {
            $this->updateChildrenEnabledStatus($menuItem, $menuItem->enabled);
            $menuItem->enabled ? MenuItemActivado::dispatch($menuItem) : MenuItemDesactivado::dispatch($menuItem);
        }

        MenuItemModificado::dispatch($menuItem);
        return response()->json($menuItem);
    }

    public function deleteMenuItem($id)
    {
        $menuItem = Menu::findOrFail($id);
        $menuRootId = $menuItem->menu_root_id;
        $parentId = $menuItem->parent_id;
        $menuItem->delete();
        $menus = Menu::where('parent_id', $parentId)->where('menu_root_id', $menuRootId)->orderBy('order', 'ASC')->get();

        foreach ($menus as $index => $menu) {
            $menu->order = $index + 1;
            $menu->save();
        }
        MenuItemEliminado::dispatch($menuItem);
        return response()->noContent();
    }

    public function getMenuRootTree()
    {
        $menusRoot = MenuRoot::all();

        return response()->json($menusRoot);
    }

    public function getMenuTree($menu_root_id)
    {
        $menus = Menu::whereNull('parent_id')
            ->where('menu_root_id', $menu_root_id)
            ->with('children')
            ->orderBy('order')
            ->get();

        $filteredMenus = $menus->filter(function ($menu) {
            return $this->filterMenuAndDescendants($menu);
        });

        $formattedMenus = $filteredMenus->map(function ($menu) {
            return $this->formatMenu($menu);
        });

        return response()->json($formattedMenus->values()->toArray());
    }

    public function getBreadcrumbs($id)
    {
        if (Config::get('menu_center.cache.enabled')) {
            $cacheKey = "menu_breadcrumbs_{$id}";
            $ttl = Config::get('menu_center.cache.ttl', 3600);
            return Cache::remember($cacheKey, $ttl, function () use ($id) {
                return $this->buildBreadcrumbsResponse($id);
            });
        }

        return $this->buildBreadcrumbsResponse($id);
    }

    private function updateChildrenEnabledStatus(Menu $menu, bool $enabledStatus)
    {
        $menu->children->each(function ($child) use ($enabledStatus) {
            $enabledStatus ? MenuActivado::dispatch($child) : MenuEliminado::dispatch($child);
            $child->update(['enabled' => $enabledStatus]);
            $this->updateChildrenEnabledStatus($child, $enabledStatus);
        });
    }

    private function formatMenu(Menu $menu): array
    {
        return [
            'id' => $menu->id,
            'parent_id' => $menu->parent_id,
            'name' => $menu->name,
            'layout' => $menu->layout,
            'route' => $menu->route,
            'icon' => $menu->icon,
            'icon_name' => $menu->icon_name,
            'order' => $menu->order,
            'description' => $menu->description,
            'separator' => $menu->separator,
            'enabled' => $menu->enabled,
            'permission_name' => $menu->permission_name,
            'created_at' => $menu->created_at->toIso8601String(),
            'updated_at' => $menu->updated_at->toIso8601String(),
            'children' => $menu->children->map(fn($child) => $this->formatMenu($child))->toArray(),
        ];
    }

    private function buildBreadcrumbsResponse($id)
    {
        $menu = Menu::with('parent')->findOrFail($id);

        if (!$this->validateBreadcrumbPermissions($menu)) {
            return response()->json(['error' => 'Forbidden'], 403);
        }
        $path = [
            [
                "name"=> "Inicio",
                "to" =>[
                    "path"=> "/"
                ]
            ]
        ];

        return response()->json(['data' =>array_merge($path, $menu->getPath())]);
    }

    private function filterMenuAndDescendants($menu)
    {
        if (!auth()->user()->can('menu-center-listar-menus') || !$this->canAccessMenu($menu)) {
            return false;
        }

        if ($menu->relationLoaded('children') && $menu->children->isNotEmpty()) {
            $menu->setRelation('children', $menu->children->filter(function ($child) use ($menu) {
                return $this->filterMenuAndDescendants($child);
            }));
        }

        return true;
    }

    private function validateBreadcrumbPermissions($menu)
    {
        while ($menu) {
            if (!$this->canAccessMenu($menu)) {
                return false;
            }
            $menu = $menu->parent; // Subir al ancestro
        }

        return true;
    }

    /**
     * Verifica si el usuario puede acceder al menú.
     */
    private function canAccessMenu($menu)
    {
        if (!$menu->permission_name) {
            return true;
        }

        return auth()->user()->can($menu->permission_name);
    }
}
