<?php

declare(strict_types=1);

namespace ILC\MenuCenter\Http\Requests;

use ILC\MenuCenter\Models\Menu;
use Illuminate\Validation\Rule;
use Illuminate\Foundation\Http\FormRequest;

class MenuRequest extends FormRequest
{
    public function authorize(): true
    {
        return true;
    }

    public function rules(): array
    {
        $menuId = $this->route('menu');

        return [
            'name' => [
                'sometimes',
                'required',
                'string',
                'max:255',
                Rule::unique('menus')
                    ->where('menu_root_id', $this->menu_root_id)
                    ->where(function ($query) {
                        if (is_null($this->parent_id)) {
                            return $query->whereNull('parent_id');
                        }
                        return $query->where('parent_id', $this->parent_id);
                    })
                    ->ignore($menuId)
            ],
            'route' => 'sometimes|required|string|min:1',
            'parent_id' => [
                'nullable',
                'integer',
                Rule::exists('menus', 'id'),
                function ($attribute, $value, $fail) {
                    $menuId = $this->route('menu');

                    if ($value === $menuId) {
                        $fail('El nodo padre no puede ser el mismo que el menú actual.');
                    }

                    if ($this->isCyclic($value, $menuId)) {
                        $fail('El nodo padre seleccionado genera un ciclo.');
                    }

                    if ($this->hasSameName($value, $this->name)) {
                        $fail('El nodo padre no puede tener el mismo nombre que el menú actual.');
                    }
                },
            ],
            'menu_root_id' => [
                'sometimes',
                'integer',
                Rule::exists('menu_root', 'id'),
            ]
        ];
    }

    protected function isCyclic($parentId, $currentId): bool
    {
        if (!$parentId || !$currentId) {
            return false;
        }

        $parent = Menu::find($parentId);

        while ($parent) {
            if ($parent->id === $currentId) {
                return true;
            }
            $parent = $parent->parent;
        }

        return false;
    }

    protected function hasSameName($parentId, $name): bool
    {
        if (!$parentId || !$name) {
            return false;
        }

        $parent = Menu::find($parentId);

        return $parent && $parent->name === $name;
    }

    public function messages(): array
    {
        return [
            'name.unique' => 'Ya existe un menú con el mismo nombre en este nivel.',
        ];
    }
}
