<?php

namespace ILC\MensajeriaCenter\Models;

use ILC\MensajeriaCenter\Database\Factories\MensajeriaCenterFactory;
use ILC\MensajeriaCenter\Helpers\DateHelper;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use OwenIt\Auditing\Auditable as AuditableTrait;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Model;

class MensajeriaCenter extends Model implements Auditable
{
    use HasFactory;
    use AuditableTrait;

    protected $table = 'mensajeria_center';

    protected $fillable = [
        'creador_id',
        'titulo',
        'descripcion_corta',
        'contenido',
        'medio',
        'roles',
        'blacklist',
        'whitelist',
        'filtros_aplicados',
        'enviar_desde',
        'terminar_hasta',
        'esta_activo',
        'posibilidadShortcut',
        'tipo_notificacion',
        'attachment',
        'attachment_name',
        'permanent_visible',
        'send_all'
    ];

    protected $casts = [
        'medio' => 'json',
        'roles' => 'array',
        'blacklist' => 'array',
        'whitelist' => 'array',
        'enviar_desde' => 'datetime',
        'terminar_hasta' => 'datetime',
        'esta_activo' => 'boolean',
    ];
    protected $appends = ['duration', 'estatus_emision'];

    public function getMedioAttribute($value)
    {
        return is_array($value) ? $value : json_decode($value, true);
    }

    public function users(): BelongsToMany
    {
        return $this->belongsToMany(
            config('mensajeria_center.user_model'),
            'mensajeria_center',
            'creador_id',
            'id'
        );
    }
    public function logs(): HasMany
    {
        return $this->hasMany(MensajeriaLog::class, 'mensajeria_id');
    }

    public function tipo_notificacion()
    {
        return $this->belongsTo(CatalogoTipoNotificaciones::class, 'tipo_notificacion');
    }

    protected static function newFactory(): MensajeriaCenterFactory
    {
        return MensajeriaCenterFactory::new();
    }

    /**
     * Obtiene el usuario que creó el mensaje.
     */
    public function create_by(): BelongsTo
    {
        return $this->belongsTo(
            config('mensajeria_center.user_model'),
            'creador_id');
    }

    //Crear atributo que me regrese el tiempo de duracion de un mensaje en minutos con el model
    public function getDurationAttribute(): string
    {
        if ($this->permanent_visible) {
            return 'Siempre visible';
        }
        return DateHelper::calculateDuration($this->enviar_desde, $this->terminar_hasta);
    }

    public function whiteListUsers()
    {
        $users = config('mensajeria_center.user_model')
            ::with(['perfil:id,user_id,nombre,primer_apellido,segundo_apellido'])
            ->whereIn('id', $this->whitelist ?? [])->select('id', 'name', 'email')
            ->get();
        $users->map(function ($user) {
            if ($user->perfil) {
                $user->full_name = $user->perfil->nombre . ' ' . $user->perfil->primer_apellido . ' ' . $user->perfil->segundo_apellido;
            } else {
                $user->full_name = $user->name ?? 'Usuario sin perfil';
            }
            return $user;
        });
        return $users;
    }

    public function blackListUsers()
    {
        $users = config('mensajeria_center.user_model')
            ::with(['perfil:id,user_id,nombre,primer_apellido,segundo_apellido'])
            ->whereIn('id', $this->blacklist ?? [])->select('id', 'name', 'email')
            ->get();
        $users->map(function ($user) {
            if ($user->perfil) {
                $user->full_name = $user->perfil->nombre . ' ' . $user->perfil->primer_apellido . ' ' . $user->perfil->segundo_apellido;
            } else {
                $user->full_name = $user->name ?? 'Usuario sin perfil';
            }
            return $user;
        });
        return $users;
    }

    public function roleList()
    {
        return config('mensajeria_center.role_model')::whereIn('id', $this->roles ?? [])->select('id', 'name')->get();

    }

    public function getEstatusEmisionAttribute(): string
    {
        $fechaActual = now();

        if ($this->permanent_visible == true) {
            return 'Permanente';
        }else {
            if ($fechaActual < $this->enviar_desde) {
                return 'Por emitir';
            } elseif ($fechaActual >= $this->enviar_desde && $fechaActual <= $this->terminar_hasta) {
                return 'Emitiendo';
            } else {
                return 'Emitido';
            }
        }
    }

}
