<?php

declare(strict_types=1);

namespace ILC\MensajeriaCenter\Http\Controllers;

use App\Models\User;
use ILC\MensajeriaCenter\Http\Requests\MensajeriaRequest;
use ILC\MensajeriaCenter\Models\MensajeriaCenter;
use ILC\MensajeriaCenter\Models\MensajeriaLog;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Response;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Storage;

class MensajeriaCenterController
{
    public static function middleware(): array
    {
        return [
            new Middleware(['auth:sanctum','can:mensajeria-center-listar'], only: ['getListMensajeria', 'getMensajeriaScope',
                                                                                    'getMensajeria', 'downloadAttachment','getUsersWhoReadMessage']),
            new Middleware(['auth:sanctum','can:mensajeria-center-crear'], only: ['createMensajeria']),
            new Middleware(['auth:sanctum','can:mensajeria-center-editar'], only: ['updateMensajeria','updateMessageStatus']),
            new Middleware(['auth:sanctum','can:mensajeria-center-eliminar'], only: ['deleteMensajeria']),

            // Middleware adicionales
            new Middleware(['auth:sanctum','can:mensajeria-center-notificacion-user'], only: ['getMessageDetailToUser', 'marcarComoLeido', 'getMensajeriaNotifications']),
        ];
    }
    public function getListMensajeria(Request $request)
    {
        $query = MensajeriaCenter::query();

        // Filter by search term
        if (!empty($request->search)) {
            $query->whereRaw('LOWER(titulo) LIKE ?', ['%' . strtolower($request->search) . '%']);
        }

        // Filter by date range
        if (!empty($request->filter_fecha_inicio) && !empty($request->filter_fecha_fin)) {
            $query->whereBetween('enviar_desde', [
                $request->filter_fecha_inicio,
                $request->filter_fecha_fin
            ]);
        }

        // Filter by notification type
        if (!empty($request->filter_tipo_notification)) {
            $query->where('tipo_notificacion', $request->filter_tipo_notification);
        }

        // Filter by status
        if (!empty($request->filter_status)) {
            $query->where('esta_activo', $request->filter_status);
        }

        // Filtrar por estatus de emisión
        if (!empty($request->filter_estatus_emision)) {
            $fechaActual = now();

            switch ($request->filter_estatus_emision) {
                case 'Por emitir':
                    $query->whereDate('enviar_desde', '>', $fechaActual);
                    break;
                case 'Emitiendo':
                    $query->whereDate('enviar_desde', '<=', $fechaActual)
                        ->whereDate('terminar_hasta', '>=', $fechaActual);
                    break;
                case 'Emitido':
                    $query->whereDate('terminar_hasta', '<', $fechaActual);
                    break;
                    case 'Permanente':
                    $query->where('permanent_visible', true);
            }
        }

        if (!empty($request->sort_by)) {
            if ($request->sort_by === 'tipo_notificacion') {
                $query->orderBy('tipo_notificacion', $request->sort_order);
            }

            $query->orderBy($request->sort_by, $request->sort_order);
        } else {
            $query->orderBy('id', 'desc');
        }

        $messages = $query->withCount(['logs' => function($query) {
                        $query->where('is_read', true);
                    }])
                    ->with(['create_by:id,name','tipo_notificacion'])
                    ->paginate($request->get('page_size', 10), ['*'], 'page', $request->get('page', 1));

        return $messages;
    }

    public function createMensajeria(MensajeriaRequest $request): JsonResponse
    {
        $user = auth()->user();
        $defaults = [
            #'medio' => json_encode($request->medio),
            'creador_id' => $user->id,
            'filtros_aplicados' => json_encode([])
        ];
        $mensajeria = MensajeriaCenter::create(array_merge($request->all(), $defaults));

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $extension = $file->getClientOriginalExtension();

            $name_hash = md5_file($file->getRealPath()). '.' . $extension;
            $filePath = $mensajeria->id. '/';

            Storage::disk('comunicaciones')->putFileAs(
                $filePath,
                $file,
                $name_hash
            );

            $mensajeria->update(
                [
                    'attachment' => $filePath . $name_hash,
                    'attachment_name' => $file->getClientOriginalName()
                ]
            );
        }
        return response()->json($mensajeria, 201);
    }

    public function getMensajeria($id): JsonResponse
    {
        $notificacion = MensajeriaCenter::with(['create_by:id,name','tipo_notificacion'])->findOrFail($id);

        $whitelist = [];
        $blacklist = [];
        $rolelist = [];

        if (is_array($notificacion->whitelist)) {
            $whitelist = $notificacion->whiteListUsers();
        }

        if (is_array($notificacion->blacklist)) {
            $blacklist = $notificacion->blackListUsers();
        }

        if (is_array($notificacion->roles)) {
            $rolelist = $notificacion->roleList();
        }

        $resultados = [
            'message_detail' => $notificacion,
            'whitelist' => $whitelist,
            'blacklist' => $blacklist,
            'rolelist' => $rolelist
        ];

        return response()->json($resultados);
    }

    public function getMessageDetailToUser($id): JsonResponse
    {
        $notificacion = MensajeriaCenter::with(['create_by:id,name','tipo_notificacion'])
                        ->where('esta_activo', true)
                        ->findOrFail($id);

        if (empty($notificacion)) {
            return response()->json(['message' => 'Comunicación inactiva'], 404);
        }

        $whitelist = [];
        $blacklist = [];
        $rolelist = [];

        if (is_array($notificacion->whitelist)) {
            $whitelist = $notificacion->whiteListUsers();
        }

        if (is_array($notificacion->blacklist)) {
            $blacklist = $notificacion->blackListUsers();
        }

        if (is_array($notificacion->roles)) {
            $rolelist = $notificacion->roleList();
        }

        $resultados = [
            'message_detail' => $notificacion,
            'whitelist' => $whitelist,
            'blacklist' => $blacklist,
            'rolelist' => $rolelist
        ];

        return response()->json($resultados);
    }

    public function updateMensajeria(MensajeriaRequest $request, $id): JsonResponse
    {
        $notificacion = MensajeriaCenter::findOrFail($id);

        $data = $request->all();

        // Handle file attachment
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $extension = $file->getClientOriginalExtension();

            $name_hash = md5_file($file->getRealPath()). '.' . $extension;
            $filePath = $notificacion->id. '/';

            Storage::disk('comunicaciones')->putFileAs(
                $filePath,
                $file,
                $name_hash
            );

            $data['attachment'] = $filePath . $name_hash;
            $data['attachment_name'] = $file->getClientOriginalName();
        }

        if (isset($data['tipo_notificacion']) && $data['tipo_notificacion'] === '[object Object]') {
            unset($data['tipo_notificacion']);
        }
        if (isset($data['roles']) && is_array($data['roles'])) {
            $data['roles'] = $data['roles'];
        }
        if (isset($data['whitelist']) && is_array($data['whitelist'])) {
            $data['whitelist'] = $data['whitelist'];
        }
        $notificacion->update($data);
        return response()->json($notificacion);
    }

    public function deleteMensajeria($id): Response
    {
        $notificacion = MensajeriaCenter::findOrFail($id);
        $notificacion->delete();

        return response()->noContent();
    }

    public function marcarComoLeido($id)
    {
        $user = auth()->user();

        $notification = MensajeriaLog::where('mensajeria_id', $id)
            ->where('users_id', $user->id)
            ->first();

        if (!$notification) {
            return response()->json(['message' => 'Comunicación no encontrado'], 404);
        }
        $notification->update(['is_read' => true]);
        return response()->json(['message' => 'Comunicación marcado como leído']);
    }

    public function getMensajeriaScope(Request $request): JsonResponse
    {
        $userModel = config('mensajeria_center.user_model');
        $roleModel = config('mensajeria_center.role_model');
        $search = null;
        if ($request->has('search')) {
            $search = $request->query('search');
        }
        $usuariosQuery = $userModel::select('id', 'name', 'email')
            ->with(['roles', 'perfil:id,user_id,nombre,primer_apellido,segundo_apellido']);


        if ($request->has('search')) {
            $search = $request->query('search');
            $usuariosQuery
                ->when($search, function ($query) use ($search) {
                    return $query->whereHas('perfil', function ($q) use ($search) {
                        $q->whereRaw('LOWER(nombre) LIKE ?', ['%' . strtolower($search) . '%'])
                            ->orWhereRaw('LOWER(primer_apellido) LIKE ?', ['%' . strtolower($search) . '%'])
                            ->orWhereRaw('LOWER(segundo_apellido) LIKE ?', ['%' . strtolower($search) . '%']);
                    });
                }
                );
        }


        if ($request->has('role')) {
            $roleFilter = $request->query('role');
            $usuariosQuery->whereHas('roles', function ($query) use ($roleFilter) {
                $query->whereRaw('LOWER(name)', 'LIKE ?', ['%' . strtolower($roleFilter) . '%']);

            });
        }

        // Filter by multiple role id
        if ($request->has('roles') && is_array($request->roles)) {
            $usuariosQuery->whereHas('roles', function ($query) use ($request) {
                $query->whereIn('id', $request->roles);
            });
        }

        // Filtrar usuarios que NO están en ciertos roles
        if ($request->has('role_exclude') && is_array($request->role_exclude)) {
            $usuariosQuery->whereDoesntHave('roles', function ($query) use ($request) {
                $query->where('id', $request->role_exclude);
            });
        }

        $usuarios = $usuariosQuery->get();

        // Transform users to include full name from profile
        $usuarios->transform(function ($user) {
            if ($user->perfil) {
                $nombreCompleto = trim($user->perfil->nombre . ' ' .
                    $user->perfil->primer_apellido . ' ' .
                    ($user->perfil->segundo_apellido ?? ''));
                $user->full_name = $nombreCompleto;
            }
            return $user;
        });

        $roles = $roleModel::select('id', 'name')->get();

        return response()->json([
            'usuarios' => $usuarios,
            'roles' => $roles,
        ]);
    }

    public function getUsersWhoReadMessage(Request $request, $id): JsonResponse
    {
        $search = null;
        if ($request->has('search')) {
            $search = $request->query('search');
        }
        $viewsUsers = MensajeriaLog::query()
            ->with(['users:id,name,email', 'users.roles:id,name', 'users.perfil:id,user_id,nombre,primer_apellido,segundo_apellido'])
            ->when($search, function ($query) use ($search) {
                return $query->whereHas('users', function ($q) use ($search) {
                    $q->whereRaw('LOWER(email) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereRaw('LOWER(name) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereHas('perfil', function ($pq) use ($search) {
                            $pq->whereRaw('LOWER(nombre) LIKE ?', ['%' . strtolower($search) . '%'])
                                ->orWhereRaw('LOWER(primer_apellido) LIKE ?', ['%' . strtolower($search) . '%'])
                                ->orWhereRaw('LOWER(segundo_apellido) LIKE ?', ['%' . strtolower($search) . '%']);
                        });
                });
            })
            ->where('mensajeria_id', $id)
            ->where('is_read', true)
            ->orderBy('updated_at', 'desc')
            ->paginate(
                $request->get('page_size', 10),
                ['*'],
                'page',
                $request->get('page', 1)
            );


        $viewsUsers->getCollection()->transform(function ($item) {
            if ($item->users && $item->users->perfil) {
                $perfil = $item->users->perfil;
                $nombreCompleto = trim($perfil->nombre . ' ' . $perfil->primer_apellido . ' ' . ($perfil->segundo_apellido ?? ''));
                $item->nombre_completo = $nombreCompleto;
            }
            return $item;
        });

        return response()->json([
            'users' => $viewsUsers,
            'message_id' => $id,
            'total_readers' => $viewsUsers->total()
        ]);
    }

    public function getMensajeriaNotifications(Request $request): JsonResponse
    {

        $user = auth()->user();
        $data = $request->all();
        $is_read =$data['is_read'] === 'true';

        $user_id = $user->id;
        $notificaciones = MensajeriaLog::query()
            ->with(['mensajeria:id,titulo,descripcion_corta,esta_activo,created_at,updated_at,tipo_notificacion', 'mensajeria.tipo_notificacion'])
            ->where('users_id', $user_id)
            ->when($is_read === false, function ($query) {
                return $query->where('is_read', false);
            })
            ->whereHas('mensajeria', function ($query) {
                $query->where('esta_activo', true);
            })
            ->orderBy('updated_at', 'desc')
            ->get();

        return response()->json($notificaciones);
    }

    public function downloadAttachment($id, $name)
    {
        try {

            $notificacion = MensajeriaCenter::findOrFail($id);
            if (empty($notificacion->attachment)) {
                return response()->json(['message' => 'Esta comunicación no contiene ningún archivo adjunto'], 404);
            }
            if (!Storage::disk('comunicaciones')->assertExists($notificacion->attachment)) {
                return response()->json(['message' => 'El archivo adjunto no se encuentra disponible en el servidor'], 404);
            }
            // Return file contents with appropriate headers
            return response(
                Storage::disk('comunicaciones')->get($notificacion->attachment),
                200,
                [
                    'Content-Type' => Storage::disk('comunicaciones')->mimeType($notificacion->attachment),
                    'Content-Disposition' => 'attachment; filename="' . $notificacion->attachment_name . '"'
                ]
            );

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json(['message' => 'Comunicación no encontrada'], 404);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error al descargar el archivo: ' . $e->getMessage()], 500);
        }
    }

    protected function userHasAccessToMessage($messageID)
    {
        $user = auth()->user();
        $userId = $user->id;
        // Check if the user has access to the message
        return MensajeriaLog::where('mensajeria_id', $messageID)
            ->where('users_id', $userId)
            ->exists();
    }

    public function updateMessageStatus(Request $request, $id): JsonResponse
    {
        try {
            $notification = MensajeriaCenter::findOrFail($id);

            // Validate the request
            $request->validate([
                'status' => 'required|boolean',
            ]);

            // Update the status directly using the boolean value from request
            $notification->update([
                'esta_activo' => $request->boolean('status')
            ]);

            return response()->json([
                'message' => 'Estatus de la comunicación actualizada correctamente',
                'notification' => $notification
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json(['message' => 'Comunicación no encontrado'], 404);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error al actualizar el estus: ' . $e->getMessage()], 500);
        }
    }

}
