<?php

declare(strict_types=1);

namespace ILC\MensajeriaCenter\Http\Controllers;

use App\Models\User;
use ILC\MensajeriaCenter\Http\Requests\MensajeriaRequest;
use ILC\MensajeriaCenter\Models\MensajeriaCenter;
use ILC\MensajeriaCenter\Models\MensajeriaLog;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Response;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\Middleware;

class MensajeriaCenterController
{
    public static function middleware(): array
    {
        return [
            new Middleware(['auth:sanctum','can:mensajeria-center-listar'], only: ['getListMensajeria', 'getMensajeriaScope', 'getMensajeria']),
            new Middleware(['auth:sanctum','can:mensajeria-center-crear'], only: ['createMensajeria']),
            new Middleware(['auth:sanctum','can:mensajeria-center-editar'], only: ['updateMensajeria']),
            new Middleware(['auth:sanctum','can:mensajeria-center-eliminar'], only: ['deleteMensajeria'])
        ];
    }
    public function getListMensajeria()
    {
        $notificacionCenterModel = MensajeriaCenter::all();

        return response()->json($notificacionCenterModel);
    }

    public function createMensajeria(MensajeriaRequest $request): JsonResponse
    {
        $user = auth()->user();
        if ($user === null) {
            $user = User::find(1);
        }

        $notificacionRequest = MensajeriaCenter::create(array_merge($request->all(), ['creador_id' => $user->id]));

        return response()->json($notificacionRequest, 201);
    }

    public function getMensajeria($id): JsonResponse
    {
        $notificacion = MensajeriaCenter::findOrFail($id);

        return response()->json($notificacion);
    }
    public function updateMensajeria(MensajeriaRequest $request, $id): JsonResponse
    {
        $notificacion = MensajeriaCenter::findOrFail($id);
        $notificacion->update($request->all());

        return response()->json($notificacion);
    }

    public function deleteMensajeria($id): Response
    {
        $notificacion = MensajeriaCenter::findOrFail($id);
        $notificacion->delete();

        return response()->noContent();
    }

    public function marcarComoLeido($id)
    {
        $user = auth()->user();

        $notification = MensajeriaLog::where('mensajeria_id', $id)
            ->where('usuario_id', $user->id)
            ->first();

        if (!$notification) {
            return response()->json(['message' => 'Notificación no encontrada'], 404);
        }

        $notification->update(['is_read' => true]);

        return response()->json(['message' => 'Notificación marcada como leída']);

    }

    public function getMensajeriaScope(Request $request): JsonResponse
    {
        $userModel = config('mensajeria_center.user_model');
        $roleModel = config('mensajeria_center.role_model');

        $usuariosQuery = $userModel::select('id', 'name', 'email')->with('roles');

        if ($request->has('search')) {
            $search = $request->query('search');
            $usuariosQuery->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                    ->orWhere('email', 'like', "%$search%");
            });
        }

        if ($request->has('role')) {
            $roleFilter = $request->query('role');
            $usuariosQuery->whereHas('roles', function ($query) use ($roleFilter) {
                $query->where('name', 'like', "%$roleFilter%");
            });
        }

        $usuarios = $usuariosQuery->get();
        $roles = $roleModel::select('id', 'name')->get();

        return response()->json([
            'usuarios' => $usuarios,
            'roles' => $roles,
        ]);
    }
}
