<?php

namespace ILC\CargaArchivos\Traits;

use ILC\CargaArchivos\Models\ILCCargaArchivo;
use Illuminate\Http\UploadedFile;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Schema;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use Doctrine\DBAL\Schema\AbstractSchemaManager;


trait HandlesFileStructure {
    protected UploadedFile $file;
    protected int $headersLineNumber;
    protected array $structureMap = [];
    protected array $modelMap = [];
    protected string $tableName = '';

    /**
     * Crea la estructura basada en la definición del archivo.
     *
     * @param $definition
     * @throws \Exception
     */
    protected function setFileDefinition($definition): void
    {
        if ($definition) {
            $fileDefinition = ILCCargaArchivo::where('referencia_nombre', $definition)->first();

            if ($fileDefinition) {
                $this->structureMap = json_decode($fileDefinition->mapa_estructura, true) ?? [];
                $this->modelMap = json_decode($fileDefinition->mapa_modelo, true) ?? [];
                $this->tableName = $this->modelMap['hojas'][0]['tabla'] ?? '';
            } else {
                throw new \Exception('No se encontró la definición del archivo con esa referencia.');
            }
        }
    }


    /**
     * Genera la estructura del archivo para guardar su definición.
     *
     * @return array
     */
    protected function buildFileStructure(): array
    {
        $fileMap = [
            'nombre' => '',
            'referencia_nombre' => '',
            'descripcion' => '',
            'mapa_estructura' => [],
            'mapa_modelo' => [],
            'tipo_archivo' => $this->file->extension() == 'xlsx' ? 'excel' : 'csv',
            'created_at' => null,
            'updated_at' => null,
        ];

        if (is_array($this->definition)) {
            $fileMap['mapa_estructura'] = $this->definition;
        } elseif ($this->definition && !is_array($this->definition)) {
            $fileMap['nombre'] = $this->definition->nombre ?? '';
            $fileMap['referencia_nombre'] = $this->definition->referencia_nombre ?? '';
            $fileMap['descripcion'] = $this->definition->descripcion ?? '';
            $fileMap['mapa_estructura'] = json_decode($this->definition->mapa_estructura ?? '{}', true);
            $fileMap['mapa_modelo'] = json_decode($this->definition->mapa_modelo ?? '{}', true);
            $fileMap['created_at'] = $this->definition->created_at ?? null;
            $fileMap['updated_at'] = $this->definition->updated_at ?? null;
        }

        return $fileMap;
    }


    /**
     * Crea la estructura del archivo desde su contenido.
     *
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @return array
     * @throws Exception
     * @throws \Exception
     */
    public function createFileStructureFromFile(UploadedFile $file, int $headersLineNumber = 1): array
    {
        try {
            $fileType = $file->getClientOriginalExtension();
            $isCSV = strtolower($fileType) === 'csv';

            if ($isCSV) {
                $fileName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $sheet = collect(array_map('str_getcsv', file($file->getPathname())));

                $rowWithHeaders = $sheet->slice($headersLineNumber - 1, 1)->first();
                if (!$rowWithHeaders) {
                    throw new Exception('Error al procesar los encabezados del archivo CSV.');
                }

                $columns = $this->processHeaders($rowWithHeaders, $headersLineNumber, $sheet);
                $tableFields = Schema::getColumnListing($fileName);
                $tableFields = array_values(
                    array_diff($tableFields, $this->getExcludedColumns($fileName))
                );

                $relatedTableFields = $this->getRelatedTablesWithFields($fileName);
                $relatedFieldsWithTables = [];

                foreach ($relatedTableFields as $relation) {
                    foreach ($relation['campos'] as $relatedField) {
                        $relatedFieldsWithTables[] = [
                            'campo' => $relatedField,
                            'tabla' => $relation['tabla'],
                        ];
                    }
                }

                $mergedFields = array_map(fn($field) => ['campo' => $field, 'tabla' => $fileName], $tableFields);
                foreach ($relatedFieldsWithTables as $relatedField) {
                    $mergedFields[] = $relatedField;
                }

                return [
                    'hojas' => [
                        [
                            'nombre' => $fileName,
                            'posicion_hoja' => 0,
                            'columnas' => $columns,
                            'campos_tabla' => $mergedFields,
                            'campos_relaciones' => $relatedTableFields,
                        ]
                    ]
                ];
            } else {
                $spreadsheet = IOFactory::load($file->getPathname());
                $sheetNames = $spreadsheet->getSheetNames();
                $sheets = Excel::toCollection(null, $this->file);

                $fileStructure = [
                    'hojas' => [],
                ];

                foreach ($sheets as $sheetIndex => $sheet) {
                    $columns = [];
                    $rowWithHeaders = $sheet->slice($headersLineNumber - 1, 1)->first();

                    $columns = $this->processHeaders($rowWithHeaders, $headersLineNumber, $sheet);

                    $tableFields = Schema::getColumnListing($sheetNames[$sheetIndex]);
                    $tableFields = array_values(
                        array_diff($tableFields, $this->getExcludedColumns($sheetNames[$sheetIndex]))
                    );
                    $relatedTableFields = $this->getRelatedTablesWithFields($sheetNames[$sheetIndex]);
                    $relatedFieldsWithTables = [];

                    foreach ($relatedTableFields as $relation) {
                        foreach ($relation['campos'] as $relatedField) {
                            $relatedFieldsWithTables[] = [
                                'campo' => $relatedField,
                                'tabla' => $relation['tabla'],
                            ];
                        }
                    }

                    $mergedFields = array_map(fn($field) => ['campo' => $field, 'tabla' => $sheetNames[$sheetIndex]],
                        $tableFields);
                    foreach ($relatedFieldsWithTables as $relatedField) {
                        $mergedFields[] = $relatedField;
                    }


                    $fileStructure['hojas'][] = [
                        'nombre' => $sheetNames[$sheetIndex],
                        'posicion_hoja' => $sheetIndex,
                        'columnas' => $columns,
                        'campos_tabla' => $mergedFields,
                        'campos_relaciones' => $relatedTableFields,
                    ];
                }
            }

            return $fileStructure;
        } catch (\Exception $e) {
            throw new \Exception("Error al crear la estructura del archivo: " . $e->getMessage());
        }
    }

    private function processHeaders($rowWithHeaders, int $headersLineNumber, $sheet): array
    {
        $columns = [];

        foreach ($rowWithHeaders as $columnIndex => $columnName) {
            $column = Coordinate::stringFromColumnIndex($columnIndex + 1);
            $row = $headersLineNumber;
            $headerColumnRow = "{$column}{$row} - {$columnName}";

            $dataFromColumn = $sheet->slice($headersLineNumber, 10)->pluck($columnIndex)->filter();
            $typeOfData = $this->determineColumnDataType($dataFromColumn);

            $columns[] = [
                'posicion' => $columnIndex,
                'nombre' => $headerColumnRow,
                'tipo' => $typeOfData,
                'requerido' => false,
            ];
        }

        return $columns;
    }




    /**
     * Trata de inferir el tipo de dato de la columna
     *
     * @param $dataFromColumn
     * @return string
     */
    protected function determineColumnDataType($dataFromColumn): string
    {
        $isBoolean = $dataFromColumn->every(fn($value) =>
            in_array(strtolower(trim($value)), ['1', '0', 'true', 'false'], true)
            );
        $isInteger = $dataFromColumn->every(fn($value) => is_numeric($value) && intval($value) == $value);
        $isDate = $dataFromColumn->every(fn($value) => strtotime($value) !== false);

        if ($isBoolean) {
            return 'boolean';
        } elseif ($isInteger) {
            return 'integer';
        } elseif ($isDate) {
            return 'date';
        } else {
            return 'string';
        }
    }


}
