<?php

namespace ILC\CargaArchivos\Traits;

use Illuminate\Support\Collection;
use Exception;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

trait DatabaseMappable {

    /**
     * @throws Exception
     */
    protected function mapRowToDatabase(Collection $row, int $rowIndex, int $sheetIndex, array $sheetStructure, array $sheetMapping): array
    {
        $mappedData = [];

        if (!empty($sheetStructure['columnas']) && !empty($sheetMapping['columnas'])) {
            $this->switchHeaderNameToTableFieldName($sheetStructure['columnas'], $sheetMapping['columnas']);

            $rules = $this->generateValidationRulesForSheet($sheetStructure);
            $messages = $this->validationMessages();

            try {
                foreach ($sheetMapping['columnas'] as $mapping) {
                    $columnaArchivo = $mapping['columna_archivo'];
                    $columnaBD = $mapping['columna_bd'] ??  null;

                    if ($columnaBD === null) {
                        Log::warning("El mapeo de columnas no contiene 'columna_bd' para la columna del archivo: $columnaArchivo.");
                        continue;
                    }

                    $sheetColumn = collect($sheetStructure['columnas'])->firstWhere('posicion', $columnaArchivo);

                    if ($sheetColumn) {
                        //$valueFromFile = $row->get($columnaBD, null); // Con Paquete Fast Excel
                        $valueFromFile = $row[$columnaArchivo] ?? null; // Con Paquete Laravel Excel
                        if ($valueFromFile !== null) {
                            $mappedData[$columnaBD] = $valueFromFile;
                        } else {
                            $mappedData[$columnaBD] = null;
                        }
                    }
                }

                $validator = Validator::make($mappedData, $rules, $messages);
                $validator->validate();

                return $mappedData;

            } catch (ValidationException $e) {
                $this->failedRegistrations++;
                $this->importDetails['records_with_errors']++;

                foreach ($e->errors() as $columnaBD => $errorMessages) {
                    $columnaArchivoIndex = array_search($columnaBD, array_column($sheetMapping['columnas'], 'columna_bd'));
                    $columnaArchivoName = $sheetStructure['columnas'][$columnaArchivoIndex]['nombre'] ?? 'desconocido';

                    $this->failedData[] = [
                        'hoja' => $sheetIndex,
                        'fila' => $rowIndex,
                        'columna' => $columnaArchivoName,
                        'errors' => $errorMessages,
                    ];

                    Log::error("Error de validación en la hoja $sheetIndex, fila $rowIndex, columna $columnaArchivoName: " . json_encode($errorMessages));
                }
            }
        } else {

            Log::error("No se encontró mapeo de columnas para la hoja con índice $sheetIndex.");
        }

        return [];
    }

    /**
     * @throws Exception
     */
    protected function switchHeaderNameToTableFieldName(array &$structureMap, array $modelMap): void
    {
        foreach ($structureMap as &$structureColumn) {
            $position = $structureColumn['posicion'];
            $modelColumn = collect($modelMap)->firstWhere('columna_archivo', $position);

            if (isset($modelColumn['columna_bd'])) {
                if ($modelColumn ) {
                    $structureColumn['nombre'] = $modelColumn['columna_bd'];
                } else {
                    throw new Exception("No se encontró coincidencia para la posición: $position en mapa_estructura.");
                }
            }
        }
    }
}
