<?php

namespace ILC\CargaArchivos\Http\Controllers;

use Exception;
use ILC\AdminUsuarios\Exceptions\AuthorizationException;
use ILC\CargaArchivos\Traits\DataLoaderQueryTrait;
use ILC\CargaArchivos\Events\FileDefinitionSaved;
use ILC\CargaArchivos\Http\Requests\DataLoaderRequest;
use ILC\CargaArchivos\Http\Requests\FileDefinitionRequest;
use ILC\CargaArchivos\Http\Requests\ImportFileRequest;
use ILC\CargaArchivos\Http\Resources\DataDefinitionResource;
use ILC\CargaArchivos\Models\ILCCargaArchivo;
use ILC\CargaArchivos\Services\CargaArchivos;
use ILC\CargaArchivos\Traits\HandlesFileStructure;
use ILC\CargaArchivos\Traits\HandlesImportDetails;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Schema;


class DataLoaderController extends Controller
{
    use HandlesImportDetails, DataLoaderQueryTrait, HandlesFileStructure, HandlesRelatedTables;

    /**
     * @OA\Get(
     *     path="/api/data-loader",
     *     operationId="indexCargaArchivos",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Obtiene el listado de cargas de archivos con filtros, búsqueda, y paginación.",
     *     description="Este endpoint permite obtener el listado de cargas de archivos con opciones de búsqueda, filtrado por tipo de archivo, y paginación.",
     *     @OA\Parameter(
     *         name="search",
     *         in="query",
     *         required=false,
     *         description="Palabra clave para buscar en los campos 'nombre' y 'descripcion'.",
     *         @OA\Schema(
     *             type="string",
     *             example="usuario"
     *         )
     *     ),
     *     @OA\Parameter(
     *          name="filetype",
     *          in="query",
     *          required=false,
     *          description="Filtra los resultados por tipo de archivo.",
     *          @OA\Schema(
     *              type="string",
     *              enum={"excel", "csv"},
     *              example="Excel"
     *          )
     *      ),
     *     @OA\Parameter(
     *         name="perPage",
     *         in="query",
     *         required=false,
     *         description="Número de elementos por página (paginación).",
     *         @OA\Schema(
     *             type="integer",
     *             example=10
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="total", type="integer", example=1),
     *                 @OA\Property(property="filter", type="integer", example=1),
     *                 @OA\Property(property="pages", type="integer", example=1),
     *                 @OA\Property(
     *                     property="data",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="id", type="integer", example=16),
     *                         @OA\Property(property="nombre", type="string", example="roles"),
     *                         @OA\Property(property="referencia_nombre", type="string", example="roles"),
     *                         @OA\Property(property="descripcion", type="string", example="los roles"),
     *                         @OA\Property(property="tipo_archivo", type="string", example="excel"),
     *                         @OA\Property(property="fecha_creacion", type="string", format="date", example="2024-12-23"),
     *                         @OA\Property(property="no_fila_encabezados", type="integer", example=5)
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Error de solicitud",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=400),
     *             @OA\Property(property="message", type="string", example="Parámetros de búsqueda o filtros inválidos.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Ocurrió un error al obtener los datos.")
     *         )
     *     )
     * )
     */


    public function index(Request $request)
    {

//        if (!$this->hasPermission('visualizar-cargas-archivos')) {
//            throw new AuthorizationException();
//        }
        $cargaArchivosQuery = $this->buildDataLoaderQuery($request);

        if ($request->filled('search')) {
            $search = $request->search;
            $cargaArchivosQuery->where(function ($query) use ($search) {
                $query->where('nombre', 'ilike', "%$search%")
                    ->orWhere('descripcion', 'ilike', "%$search%");
            });
        }

        if ($request->filled('filetype')) {
            $fileType = $request->filetype;
            $cargaArchivosQuery->where(function ($query) use ($fileType) {
                $query->where('tipo_archivo', $fileType);
            });
        }

        $this->applySorting($cargaArchivosQuery, $request);
        $perPage = $request->get('perPage', 10);
        $cargasArchivos = $cargaArchivosQuery->paginate($perPage);

        return $this->sendResponse([
            'total' => $cargasArchivos->total(),
            'filter' => $cargasArchivos->total(),
            'pages' => $cargasArchivos->lastPage(),
            'data' => $cargasArchivos->items()
        ]);
    }


    /**
     * @OA\Post(
     *     path="/api/data-loader/process",
     *     operationId="processCargaArchivos",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Procesa un archivo y obtiene su estructura.",
     *     description="Este endpoint permite cargar un archivo, procesarlo y devolver su estructura (por ejemplo, las cabeceras de un CSV o Excel).",
     *     @OA\RequestBody(
     *         required=true,
     *         description="Archivo para procesar.",
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"archivo"},
     *                 @OA\Property(
     *                     property="archivo",
     *                     type="string",
     *                     format="binary",
     *                     description="Archivo que será procesado."
     *                 ),
     *                 @OA\Property(
     *                     property="no_fila_encabezados",
     *                     type="integer",
     *                     description="Número de la fila que contiene los encabezados del archivo (opcional, por defecto 1).",
     *                     example=1
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(
     *                     property="data",
     *                     type="object",
     *                     @OA\Property(
     *                         property="hojas",
     *                         type="array",
     *                         @OA\Items(
     *                             type="object",
     *                             @OA\Property(property="nombre", type="string", example="roles"),
     *                             @OA\Property(property="posicion_hoja", type="integer", example=0),
     *                             @OA\Property(
     *                                 property="columnas",
     *                                 type="array",
     *                                 @OA\Items(
     *                                     type="object",
     *                                     @OA\Property(property="posicion", type="integer", example=0),
     *                                     @OA\Property(property="nombre", type="string", example="A1 - id"),
     *                                     @OA\Property(property="tipo", type="string", example="boolean"),
     *                                     @OA\Property(property="requerido", type="boolean", example=false)
     *                                 )
     *                             ),
     *                             @OA\Property(
     *                                 property="campos_tabla",
     *                                 type="array",
     *                                 @OA\Items(
     *                                     type="object",
     *                                     @OA\Property(property="campo", type="string", example="name"),
     *                                     @OA\Property(property="tabla", type="string", example="roles")
     *                                 )
     *                             ),
     *                             @OA\Property(
     *                                 property="campos_relaciones",
     *                                 type="array",
     *                                 @OA\Items(type="object")
     *                             )
     *                         )
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Solicitud incorrecta",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=422),
     *             @OA\Property(property="message", type="string", example="El archivo subido debe tener extensión xlsx o csv.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Se produjo un error al procesar el archivo.")
     *         )
     *     )
     * )
     */

    public function process(DataLoaderRequest $request): JsonResponse
    {
        try {
            $file = $request->file('archivo');
            $fileType = $file->getClientOriginalExtension();
            $headersLineNumber = strtolower($fileType) === 'csv' ? 1 : $request->input('no_fila_encabezados', 1);
            $fileStructure = CargaArchivos::getFileStructure($file, $headersLineNumber);

            if ($fileStructure ) {
                return $this->sendResponse([ 'data' =>  $fileStructure ]);
            }

            return $this->sendError('No se pudo procesar el archivo.');
        }  catch (\Exception $e) {
            return $this->sendError(
                'Se produjo un error al procesar el archivo: ' . $e->getMessage(),
                500
            );
        }

    }


    /**
     * @OA\Post(
     *     path="/api/data-loader/save",
     *     operationId="saveCargaArchivos",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Guarda una nueva definición de archivo.",
     *     description="Este endpoint permite guardar una nueva definición de archivo. El archivo será procesado y guardado, y un evento será disparado cuando se guarde correctamente.",
     *     @OA\RequestBody(
     *         required=true,
     *         description="Datos de la definición de archivo a guardar.",
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"nombre", "referencia_nombre", "descripcion", "tipo_archivo", "mapa_estructura", "mapa_modelo"},
     *                 @OA\Property(property="nombre", type="string", example="roles", description="Nombre de la definición del archivo."),
     *                 @OA\Property(property="referencia_nombre", type="string", example="roles", description="Referencia única para la definición del archivo."),
     *                 @OA\Property(property="no_fila_encabezados", type="integer", example=5, description="Número de la fila que contiene los encabezados del archivo."),
     *                 @OA\Property(property="descripcion", type="string", example="The roles", description="Descripción de la definición del archivo."),
     *                 @OA\Property(property="tipo_archivo", type="string", example="excel", description="Tipo de archivo (e.g., 'excel', 'csv')."),
     *                 @OA\Property(
     *                     property="mapa_estructura",
     *                     type="object",
     *                     @OA\Property(
     *                         property="hojas",
     *                         type="array",
     *                         @OA\Items(
     *                             type="object",
     *                             @OA\Property(property="nombre", type="string", example="roles", description="Nombre de la hoja."),
     *                             @OA\Property(property="posicion_hoja", type="integer", example=0, description="Posición de la hoja en el archivo."),
     *                             @OA\Property(
     *                                 property="columnas",
     *                                 type="array",
     *                                 @OA\Items(
     *                                     type="object",
     *                                     @OA\Property(property="posicion", type="integer", example=0, description="Posición de la columna en la hoja."),
     *                                     @OA\Property(property="nombre", type="string", example="Rol", description="Nombre de la columna."),
     *                                     @OA\Property(property="tipo", type="string", example="alfábetico", description="Tipo de dato de la columna."),
     *                                     @OA\Property(property="requerido", type="boolean", example=true, description="Si la columna es requerida."),
     *                                     @OA\Property(property="registro_unico", type="boolean", example=true, description="Si la columna es única en los registros.")
     *                                 )
     *                             )
     *                         )
     *                     )
     *                 ),
     *                 @OA\Property(
     *                     property="mapa_modelo",
     *                     type="object",
     *                     @OA\Property(
     *                         property="hojas",
     *                         type="array",
     *                         @OA\Items(
     *                             type="object",
     *                             @OA\Property(property="nombre", type="string", example="roles", description="Nombre de la hoja."),
     *                             @OA\Property(property="tabla", type="string", example="roles", description="Tabla a la que se mapea la hoja."),
     *                             @OA\Property(
     *                                 property="columnas",
     *                                 type="array",
     *                                 @OA\Items(
     *                                     type="object",
     *                                     @OA\Property(property="columna_archivo", type="integer", example=0, description="Índice de la columna en el archivo."),
     *                                     @OA\Property(property="columna_bd", type="string", example="name", description="Nombre de la columna en la base de datos.")
     *                                 )
     *                             )
     *                         )
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Definición de archivo guardada exitosamente",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="Definición de archivo guardada exitosamente.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Solicitud incorrecta",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=422),
     *             @OA\Property(property="message", type="string", example="El campo nombre de referencia debe contener un valor único.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Se produjo un error al procesar el archivo.")
     *         )
     *     )
     * )
     */

    public function save(FileDefinitionRequest $request)
    {
        $validatedData = $request->validated();
        $fileDefinition = new ILCCargaArchivo();
        $this->createFileDefinition($validatedData, $fileDefinition);

        if ($fileDefinition->save()) {

            event(new FileDefinitionSaved($fileDefinition));

            return $this->sendResponse(['message' => 'Definición de archivo guardada exitosamente.']);
        }

        return $this->sendError('No se pudo guardar el archivo.');
    }

    /**
     * @OA\Put(
     *     path="/api/data-loader/{id}",
     *     operationId="updateDataLoader",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Actualiza una definición de archivo existente.",
     *     description="Este endpoint permite actualizar una definición de archivo por su ID. El archivo será validado y actualizado con los nuevos datos.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición del archivo a actualizar.",
     *         @OA\Schema(
     *             type="integer",
     *             example=18
     *         )
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         description="Datos de la definición de archivo a actualizar.",
     *         @OA\JsonContent(
     *             type="object",
     *             required={"nombre", "referencia_nombre", "no_fila_encabezados", "descripcion", "mapa_estructura", "mapa_modelo", "tipo_archivo"},
     *             @OA\Property(property="nombre", type="string", example="roles"),
     *             @OA\Property(property="referencia_nombre", type="string", example="roles"),
     *             @OA\Property(property="no_fila_encabezados", type="integer", example=5),
     *             @OA\Property(property="descripcion", type="string", example="The roles"),
     *             @OA\Property(property="mapa_estructura", type="object",
     *                 @OA\Property(property="hojas", type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="nombre", type="string", example="roles"),
     *                         @OA\Property(property="posicion_hoja", type="integer", example=0),
     *                         @OA\Property(property="columnas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="posicion", type="integer", example=0),
     *                                 @OA\Property(property="nombre", type="string", example="Rol"),
     *                                 @OA\Property(property="tipo", type="string", example="alfábetico"),
     *                                 @OA\Property(property="requerido", type="boolean", example=true),
     *                                 @OA\Property(property="registro_unico", type="boolean", example=true)
     *                             )
     *                         )
     *                     )
     *                 )
     *             ),
     *             @OA\Property(property="mapa_modelo", type="object",
     *                 @OA\Property(property="hojas", type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="nombre", type="string", example="roles"),
     *                         @OA\Property(property="tabla", type="string", example="roles"),
     *                         @OA\Property(property="columnas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="columna_archivo", type="integer", example=0),
     *                                 @OA\Property(property="columna_bd", type="string", example="name")
     *                             )
     *                         )
     *                     )
     *                 )
     *             ),
     *             @OA\Property(property="tipo_archivo", type="string", example="excel"),
     *             @OA\Property(property="created_at", type="string", format="nullable", example=null),
     *             @OA\Property(property="updated_at", type="string", format="nullable", example=null)
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Definición de archivo actualizada exitosamente",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="Definición de archivo actualizada exitosamente.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición de archivo",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=404),
     *             @OA\Property(property="message", type="string", example="No se encontró la definición del archivo.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Error de validación o datos inválidos",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=400),
     *             @OA\Property(property="message", type="string", example="No se pudo actualizar la definición del archivo.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Ha ocurrido un error al intentar actualizar la definición del archivo.")
     *         )
     *     )
     * )
     */

    public function update(FileDefinitionRequest $request, int $id): JsonResponse
    {
        $fileDefinition = ILCCargaArchivo::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición del archivo.', 404);
        }

        try {
            $validatedData = $request->validated();
            $this->createFileDefinition($validatedData, $fileDefinition);

            if ($fileDefinition->save()) {
                event(new FileDefinitionSaved($fileDefinition));

                return $this->sendResponse(['message' => 'Definición de archivo actualizada exitosamente.']);
            }

            return $this->sendError('No se pudo actualizar la definición del archivo.');
        } catch (Exception $e) {
            Log::error('Error al actualizar la definición del archivo: ' . $e->getMessage());

            return $this->sendError('Ha ocurrido un error al intentar actualizar la definición del archivo.', 500);
        }
    }
    /**
     * @OA\Post(
     *     path="/api/data-loader/import",
     *     operationId="importDataLoader",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Importa un archivo y procesa su contenido.",
     *     description="Este endpoint permite importar un archivo, procesarlo y almacenar los resultados. El archivo será validado y los datos serán almacenados según la referencia proporcionada.",
     *     @OA\RequestBody(
     *         required=true,
     *         description="Datos para importar el archivo.",
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"referencia_nombre", "archivo"},
     *                 @OA\Property(property="referencia_nombre", type="string", example="usuarios", description="Referencia que determina cómo se debe procesar el archivo."),
     *                 @OA\Property(
     *                     property="archivo",
     *                     type="string",
     *                     format="binary",
     *                     description="Archivo que será procesado e importado."
     *                 ),
     *                 @OA\Property(
     *                     property="no_fila_encabezados",
     *                     type="integer",
     *                     description="Número de la fila que contiene los encabezados del archivo (opcional, por defecto 1).",
     *                     example=1
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Archivo procesado e importado exitosamente",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Se ha terminado de procesar el archivo."),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="success", type="boolean", example=true),
     *                 @OA\Property(property="importDetails", type="object",
     *                     @OA\Property(property="file_name", type="string", example="roles.xlsx"),
     *                     @OA\Property(property="total_records", type="integer", example=4),
     *                     @OA\Property(property="records_added", type="integer", example=0),
     *                     @OA\Property(property="records_updated", type="integer", example=4),
     *                     @OA\Property(property="records_with_errors", type="integer", example=0),
     *                     @OA\Property(property="errors_detail", type="array",
     *                         @OA\Items(type="string")
     *                     )
     *                 ),
     *                 @OA\Property(property="message", type="string", example="Los registros se han guardado correctamente.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Archivo procesado e importado con errores de validación",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Se ha terminado de procesar el archivo."),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="success", type="boolean", example=true),
     *                 @OA\Property(property="importDetails", type="object",
     *                     @OA\Property(property="file_name", type="string", example="roles_errores.xlsx"),
     *                     @OA\Property(property="total_records", type="integer", example=3),
     *                     @OA\Property(property="records_added", type="integer", example=1),
     *                     @OA\Property(property="records_updated", type="integer", example=2),
     *                     @OA\Property(property="records_with_errors", type="integer", example=3),
     *                     @OA\Property(property="errors_detail", type="array",
     *                         @OA\Items(
     *                             type="object",
     *                             @OA\Property(property="hoja", type="integer", example=0),
     *                             @OA\Property(property="fila", type="integer", example=7),
     *                             @OA\Property(property="columna", type="string", example="guard_name"),
     *                             @OA\Property(property="errors", type="array",
     *                                 @OA\Items(type="string", example="El campo guard name es requerido.")
     *                             )
     *                         )
     *                     )
     *                 ),
     *                 @OA\Property(property="message", type="string", example="Hubo 3 errores durante la importación.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="No se pudieron importar los datos del archivo."),
     *             @OA\Property(property="error", type="string", example="Detalles del error.")
     *         )
     *     )
     * )
     */

    public function import(ImportFileRequest $request): JsonResponse
    {
        try {
            $referencia = $request->input('referencia_nombre');
            $file = $request->file('archivo');

            $headersLineNumber = $request->input('no_fila_encabezados', 1);
            $import = CargaArchivos::archivo($referencia, $file, $headersLineNumber)->store();

            return response()->json([
                'message' => 'Se ha terminado de procesar el archivo.',
                'result' => $import,
            ]);

        } catch (Exception $e) {

            Log::error('Error en la importación: ' . $e->getMessage());

            return response()->json([
                'message' => 'No se pudieron importar los datos del archivo.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    /**
     * @OA\Get(
     *     path="/api/data-loader/{id}",
     *     operationId="showDataLoader",
     *     tags={"Definiciones de Cargas"},
     *     summary="Obtiene una definición de archivo por su ID.",
     *     security={{"BearerAuth":{}}},
     *     description="Este endpoint permite obtener una definición de archivo existente por su ID, procesar la estructura y los campos de las tablas relacionadas.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición del archivo.",
     *         @OA\Schema(
     *             type="integer",
     *             example=18
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="data", type="object",
     *                     @OA\Property(property="id", type="integer", example=18),
     *                     @OA\Property(property="nombre", type="string", example="roles"),
     *                     @OA\Property(property="referencia_nombre", type="string", example="roles"),
     *                     @OA\Property(property="no_fila_encabezados", type="integer", example=5),
     *                     @OA\Property(property="descripcion", type="string", example="The roles edited"),
     *                     @OA\Property(property="mapa_estructura", type="object",
     *                         @OA\Property(property="hojas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="nombre", type="string", example="roles"),
     *                                 @OA\Property(property="columnas", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="tipo", type="string", example="alfábetico"),
     *                                         @OA\Property(property="nombre", type="string", example="Rol"),
     *                                         @OA\Property(property="posicion", type="integer", example=0),
     *                                         @OA\Property(property="requerido", type="boolean", example=true),
     *                                         @OA\Property(property="registro_unico", type="boolean", example=true)
     *                                     )
     *                                 ),
     *                                 @OA\Property(property="posicion_hoja", type="integer", example=0),
     *                                 @OA\Property(property="campos_tabla", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="campo", type="string", example="name"),
     *                                         @OA\Property(property="tabla", type="string", example="roles")
     *                                     )
     *                                 ),
     *                                 @OA\Property(property="campos_relaciones", type="array",
     *                                     @OA\Items(type="object")
     *                                 )
     *                             )
     *                         )
     *                     ),
     *                     @OA\Property(property="mapa_modelo", type="object",
     *                         @OA\Property(property="hojas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="tabla", type="string", example="roles"),
     *                                 @OA\Property(property="nombre", type="string", example="roles"),
     *                                 @OA\Property(property="columnas", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="columna_bd", type="string", example="name"),
     *                                         @OA\Property(property="columna_archivo", type="integer", example=0)
     *                                     )
     *                                 )
     *                             )
     *                         )
     *                     ),
     *                     @OA\Property(property="tipo_archivo", type="string", example="excel"),
     *                     @OA\Property(property="created_at", type="string", format="date-time", example="2025-01-01T23:46:39.000000Z"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time", example="2025-01-02T00:11:24.000000Z")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición del archivo",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="No se encontró la definición del archivo.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="Ocurrió un error al obtener la definición del archivo.")
     *         )
     *     )
     * )
     */

    public function show(int $id): JsonResponse
    {
        $fileDefinition = ILCCargaArchivo::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición del archivo.', 404);
        }

        try {
            $modelMap = json_decode($fileDefinition->mapa_modelo);
            $structureMap = json_decode($fileDefinition->mapa_estructura);
            foreach ($modelMap->hojas as $index => $hoja) {
                $tableFields = Schema::getColumnListing($hoja->tabla);
                $tableFields = array_values(array_diff($tableFields, $this->getExcludedFields($hoja->tabla)));

                $currentTableFields = array_map(fn($field) => [
                    'campo' => $field,
                    'tabla' => $hoja->tabla,
                ], $tableFields);

                $relatedTableFields = $this->getRelatedTablesWithFields($hoja->tabla);
                $relatedFieldsWithTables = [];

                foreach ($relatedTableFields as $relation) {
                    foreach ($relation['campos'] as $relatedField) {
                        $relatedFieldsWithTables[] = [
                            'campo' => $relatedField,
                            'tabla' => $relation['tabla'],
                        ];
                    }
                }

                $mergedFields = array_merge($currentTableFields, $relatedFieldsWithTables);
                $structureMap->hojas[$index]->campos_tabla = $mergedFields;
                $structureMap->hojas[$index]->campos_relaciones = $relatedTableFields;
            }

            $response = [
                'id' => $fileDefinition->id,
                'nombre' => $fileDefinition->nombre,
                'referencia_nombre' => $fileDefinition->referencia_nombre,
                'no_fila_encabezados' => $fileDefinition->no_fila_encabezados,
                'descripcion' => $fileDefinition->descripcion,
                'mapa_estructura' => json_decode(json_encode($structureMap), true),
                'mapa_modelo' => json_decode($fileDefinition->mapa_modelo, true),
                'tipo_archivo' => $fileDefinition->tipo_archivo,
                'created_at' => $fileDefinition->created_at,
                'updated_at' => $fileDefinition->updated_at,
            ];

            return $this->sendResponse([ 'data' => $response ]);
        } catch (Exception $e) {
            Log::error('Error al obtener la definición del archivo: ' . $e->getMessage());

            return $this->sendError('Ocurrió un error al obtener la definición del archivo.', 500);
        }
    }


    /**
     * @OA\Delete(
     *     path="/api/data-loader/{id}",
     *     operationId="destroyDataLoader",
     *     tags={"Definiciones de Cargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Elimina una definición de archivo por su ID.",
     *     description="Este endpoint permite eliminar una definición de archivo existente por su ID. Si no se encuentra la definición, se devuelve un error.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición del archivo a eliminar.",
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Definición de archivo eliminada exitosamente",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="La definición del archivo se ha eliminado exitosamente.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición del archivo",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=400),
     *             @OA\Property(property="message", type="string", example="No se encontró la definición del archivo.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Ha ocurrido un error al intentar eliminar la definición del archivo.")
     *         )
     *     )
     * )
     */

    public function destroy(int $id): JsonResponse
    {
        $fileDefinition = ILCCargaArchivo::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición del archivo.');
        }

        try {

            $fileDefinition->delete();

            return $this->sendResponse('La definición del archivo se ha eliminado exitosamente.');
        } catch (Exception $e) {

            Log::error('Error al eliminar definición de archivo: '.$e->getMessage());

            return $this->sendError('Ha ocurrido un error al intentar eliminar la definición del archivo.');
        }
    }

    /**
     * Obtiene las tablas relacionadas y los campos de una tabla específica.
     *
     * @param Request $request
     * @param string $tableName
     * @return JsonResponse
     */

    public function getRelatedData(string $table): JsonResponse
    {
        try {
            $relatedTables = $this->getRelatedTablesWithFields($table);

            return response()->json([
                'success' => true,
                'data' => $relatedTables,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * @param $validatedData
     * @param ILCCargaArchivo $fileDefinition
     * @return void
     */
    public function createFileDefinition($validatedData, ILCCargaArchivo $fileDefinition): void
    {
        $fileDefinition->nombre = $validatedData['nombre'];
        $fileDefinition->referencia_nombre = Str::slug($validatedData['nombre']);
        $fileDefinition->no_fila_encabezados = $validatedData['no_fila_encabezados'];
        $fileDefinition->descripcion = $validatedData['descripcion'] ?? null;
        $fileDefinition->mapa_estructura = json_encode($validatedData['mapa_estructura']);
        $fileDefinition->mapa_modelo = json_encode($validatedData['mapa_modelo']);
        $fileDefinition->tipo_archivo = $validatedData['tipo_archivo'];
    }

}
