<?php

namespace ILC\CargaArchivos\Traits;

use Illuminate\Support\Facades\Log;

trait DatabaseMappable
{
    /**
     * Mapea una fila del archivo Excel a las columnas de la tabla
     *
     * @param array $row
     * @param int $rowIndex
     * @param int $sheetIndex
     * @param array $sheetStructure
     * @param array $sheetMapping
     * @return array|null
     */
    protected function mapRowToDatabase(array $row, int $rowIndex, int $sheetIndex, array $sheetStructure, array $sheetMapping): ?array
    {
        try {
            if ($this->isEmptyRow($row)) {
                return null;
            }

            $mappedRow = [];
            $columnMappings = $sheetMapping['columnas'] ?? [];

            foreach ($columnMappings as $mapping) {
                $fileColumn = $mapping['columna_archivo'] ?? null;
                $dbColumn = $mapping['columna_bd'] ?? null;

                if ($fileColumn === null || $dbColumn === null) {
                    continue;
                }

                if (is_numeric($fileColumn)) {
                    $columnKeys = array_keys($row);
                    $fileColumnName = $columnKeys[$fileColumn] ?? null;

                    if ($fileColumnName !== null && isset($row[$fileColumnName])) {
                        $mappedRow[$dbColumn] = $row[$fileColumnName];
                    }
                } else {
                    if (isset($row[$fileColumn])) {
                        $mappedRow[$dbColumn] = $row[$fileColumn];
                    }
                }
            }

            $uniqueField = null;
            foreach ($sheetStructure['columnas'] as $column) {
                if (isset($column['registro_unico']) && $column['registro_unico'] === true) {
                    foreach ($columnMappings as $mapping) {
                        if ($column['posicion'] === $mapping['columna_archivo'] && isset($mapping['columna_bd'])) {
                            $uniqueField = $mapping['columna_bd'];
                            break 2;
                        }
                    }
                }
            }

            if ($uniqueField && (!isset($mappedRow[$uniqueField]) || $mappedRow[$uniqueField] === null || $mappedRow[$uniqueField] === '')) {
                Log::warning("Fila {$rowIndex} ignorada: campo único '{$uniqueField}' no encontrado o vacío.");
                return null;
            }

            if (method_exists($this, 'generateValidationRulesForSheet')) {
                try {
                    $validationRules = $this->generateValidationRulesForSheet($sheetStructure);
                    $this->validateMappedData($mappedRow, $validationRules, $this->validationMessages());
                } catch (\Illuminate\Validation\ValidationException $e) {
                    $errors = $e->errors();
                    $errorMessages = [];

                    foreach ($errors as $field => $messages) {
                        foreach ($messages as $message) {
                            $errorMessages[] = "Campo '{$field}': {$message}";
                        }
                    }

                    if (property_exists($this, 'failedData')) {
                        $this->failedData[] = [
                            'hoja' => $this->currentSheetName ?? 'No disponible',
                            'fila' => $rowIndex,
                            'errors' => $errorMessages
                        ];
                    }

                    Log::warning("Validación fallida para fila {$rowIndex}: " . implode(', ', $errorMessages));
                    return null;
                }
            }

            return $mappedRow;
        } catch (\Exception $e) {
            Log::error("Error al mapear fila {$rowIndex} de la hoja {$sheetIndex}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Revisa si la fila está vacía
     *
     * @param array $row
     * @return bool
     */
    protected function isEmptyRow(array $row): bool
    {
        foreach ($row as $value) {
            if ($value !== null && $value !== '') {
                return false;
            }
        }
        return true;
    }
}
