<?php

namespace ILC\CargaArchivos\Http\Requests;
use Illuminate\Support\Str;

class DataLoaderRequest extends FormRequest
{
    protected $stopOnFirstFailure = true;

    public function authorize(): bool
    {
        return true;
    }
    
    public function rules(): array
    {
        $maxUploadFileSize = $this->getMaxUploadFileSize();

        return [
            'archivo' => "required|file|mimes:xlsx,csv|max:{$maxUploadFileSize}",
        ];
    }

    public function messages(): array
    {
        return [
            'archivo.required' => 'Favor de incluir el archivo.',
            'archivo.file' => 'Debes subir un archivo válido.',
            'archivo.mimes' => 'El archivo subido debe tener la extensión xlsx o csv.',
            'archivo.max' => 'El archivo subido debe ser menor de ' .$this->getMaxUploadFileSizeFromConfigOptions() . '.',
        ];
    }

    /**
     * Obtiene el valor de las subida de archivos de php.ini convertido a kilobytes
     *
     * @return int
     */
    public function getMaxUploadFileSize(): int
    {
        $configOptionMaxFileSize = ini_get('upload_max_filesize');
        return $this->convertConfigOptionSizeToBytes($configOptionMaxFileSize) / 1024;
    }

    /**
     * Convierte el valor especificado de string
     * (ej. "2M", "512K" a kilobytes para usarse en la validación
     *
     * @param string $size
     * @return int
     */
    protected function convertConfigOptionSizeToBytes(string $size): int
    {
        $unit = strtolower(substr($size, -1));
        $size = (int)$size;

        switch ($unit) {
            case 'g':
                $size *= 1024;
            case 'm':
                $size *= 1024;
            case 'k':
                $size *= 1024;
        }

        return $size;
    }

    /**
     * Retorna el valor almacenado en al archivo php.ini para subida de archivos
     *
     * @return string
     */
    protected function getMaxUploadFileSizeFromConfigOptions(): string
    {
        return ini_get('upload_max_filesize');
    }
}
