<?php

/**
 * @license Apache 2.0
 */

namespace ILC\CargaArchivos\Http\Controllers;

use Exception;
use ILC\CargaArchivos\Exceptions\AuthorizationException;
use ILC\CargaArchivos\Exports\DataExportService;
use ILC\CargaArchivos\Http\Requests\TemplateDefinitionRequest;
use ILC\CargaArchivos\Models\ILCPlantillaDescarga;
use ILC\CargaArchivos\Traits\DataExporterQueryTrait;
use ILC\CargaArchivos\Traits\HandlesFileStructure;
use ILC\CargaArchivos\Traits\HandlesImportDetails;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use Illuminate\Database\QueryException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Rap2hpoutre\FastExcel\FastExcel;
use Rap2hpoutre\FastExcel\SheetCollection;
use Illuminate\Support\Facades\Config;
use Carbon\Carbon;

class DataExporterController extends Controller
{
    use HandlesImportDetails, DataExporterQueryTrait, HandlesFileStructure, HandlesRelatedTables;

    protected DataExportService $exporterService;
    public function __construct(DataExportService $exporterService)
    {
        $this->exporterService = $exporterService;
    }


    /**
     * @OA\Get(
     *     path="/api/data-exporter",
     *     operationId="indexDataExporter",
     *     tags={"Definiciones de descargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Obtiene el listado de cargas de archivos.",
     *     description="Retorna un listado de las definiciones de descargas de archivo que cumplan con los parámetros solicitados (filtros, búsqueda, no. de elementos en la paginación).",
     *     @OA\Parameter(
     *         name="filter",
     *         in="query",
     *         required=false,
     *         description="Filtro para las definiciones de descargas de archivos (por ejemplo, filtrar por tipo de archivo).",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="search",
     *         in="query",
     *         required=false,
     *         description="Palabra clave para buscar en los registros de las definiciones de descargas de archivos.",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         required=false,
     *         description="Número de página para la paginación.",
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="total",
     *         in="query",
     *         required=false,
     *         description="Número total de elementos por página (paginación).",
     *         @OA\Schema(
     *             type="integer",
     *             example=10
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="total", type="integer", example=4),
     *                 @OA\Property(property="filter", type="integer", example=4),
     *                 @OA\Property(property="pages", type="integer", example=1),
     *                 @OA\Property(
     *                     property="data",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="id", type="integer", example=38),
     *                         @OA\Property(property="nombre", type="string", example="users"),
     *                         @OA\Property(property="referencia_nombre", type="string", example="users"),
     *                         @OA\Property(property="descripcion", type="string", example="la tabla users"),
     *                         @OA\Property(property="fecha_creacion", type="string", format="date", example="2024-12-27")
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Not found"
     *     )
     * )
     * @throws AuthorizationException
     */


    public function index(Request $request): JsonResponse
    {
        if (!$this->hasPermission('visualizar-definiciones-exportacion')) {
            throw new AuthorizationException();
        }
        $exportaArchivosQuery = $this->buildDataExporterQuery($request);

        if ($request->filled('search')) {
            $search = $request->search;
            $exportaArchivosQuery->where(function ($query) use ($search) {
                $query->where('nombre', 'ilike', "%$search%")
                    ->orWhere('descripcion', 'ilike', "%$search%");
            });
        }

        $this->applySorting($exportaArchivosQuery, $request);
        $perPage = $request->get('perPage', 10);
        $exportaArchivos = $exportaArchivosQuery->paginate($perPage);

        return $this->sendResponse([
            'total' => $exportaArchivos->total(),
            'filter' => $exportaArchivos->total(),
            'pages' => $exportaArchivos->lastPage(),
            'data' => $exportaArchivos->items()
        ]);
    }

    /**
     * @OA\Post(
     *     path="/api/data-exporter/save",
     *     operationId="saveDataExporter",
     *     tags={"Definiciones de descargas"},
     *     summary="Almacena la definición de una plantilla para descargas de datos.",
     *     security={{"BearerAuth":{}}},
     *     description="Este endpoint permite almacenar una nueva definición de plantilla para descargas de datos.",
     *     @OA\RequestBody(
     *         required=true,
     *         description="Datos de la plantilla para ser almacenados.",
     *         @OA\JsonContent(
     *             required={"nombre", "mapa_modelo"},
     *             type="object",
     *             @OA\Property(property="nombre", type="string", example="Plantilla de usuarios", description="Nombre de la plantilla."),
     *             @OA\Property(property="descripcion", type="string", example="Plantilla para la descarga de datos de usuarios.", description="Descripción de la plantilla."),
     *             @OA\Property(property="mapa_modelo", type="object",
     *                 description="Mapa de los campos del modelo.",
     *                 @OA\Property(property="tablas", type="array",
     *                     @OA\Items(type="object",
     *                         @OA\Property(property="tabla", type="string", example="usuarios", description="Nombre de la tabla o modelo."),
     *                         @OA\Property(property="isEloquent", type="boolean", example=false, description="Determina si se construirá mediante Eloquent Builder (true) o Query Builder (false). Por defecto: false."),
     *                         @OA\Property(property="campos", type="array",
     *                             description="Lista de campos de la tabla.",
     *                             @OA\Items(type="object",
     *                                 @OA\Property(property="nombre", type="string", example="nombre", description="Nombre del campo."),
     *                                 @OA\Property(property="apodo", type="string", example="Nombre", description="Nombre del campo como aparecerá en la plantilla."),
     *                                 @OA\Property(property="tipo", type="string", example="alfabetico", description="Tipo del campo.", enum={"numerico", "alfabetico", "fecha", "verdadero/falso"}),
     *                                 @OA\Property(property="requerido", type="boolean", example=true, description="Indica si el campo es requerido."),
     *                                 @OA\Property(property="forTemplate", type="boolean", example=false, description="Indica si el campo está disponible para la plantilla.")
     *                             )
     *                         )
     *                     )
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="message", type="string", example="Se ha registrado la definición de la plantilla.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Error de validación o nombre duplicado",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=422),
     *             @OA\Property(property="message", type="string", example="Error de validación: ...")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error inesperado o de base de datos",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="message", type="string", example="Ocurrió un error inesperado.")
     *         )
     *     )
     * )
     * @throws AuthorizationException
     */

    public function save(TemplateDefinitionRequest $request): JsonResponse
    {
        if (!$this->hasPermission('guardar-definiciones-exportacion')) {
            throw new AuthorizationException();
        }
        try {
            $validatedData = $request->validated();

            ILCPlantillaDescarga::create([
                'nombre' => $validatedData['nombre'],
                'referencia_nombre' => Str::slug($validatedData['nombre']),
                'descripcion' => $validatedData['descripcion'],
                'mapa_modelo' => $validatedData['mapa_modelo'],
            ]);

            return $this->sendResponse([
                'message' => 'Se ha registrado la definición de la plantilla.'
            ]);
        } catch (ValidationException $e) {
            return $this->sendError('Error de validación: '.$e->errors(), 422);

        } catch (QueryException $e) {
            if ($e->getCode() === '23505') { // Código de Error de PostgreSQL "unique constraint"
                return $this->sendError('El nombre de la definición debe ser único, ya existe un registro con ese valor. ', 422);
            }

            return $this->sendError('Ocurrió un error en la base de datos.', 500);
        } catch (Exception $e) {
            return $this->sendError('Ocurrió un error inesperado.', 500);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/data-exporter/{id}",
     *     operationId="showDataExporter",
     *     tags={"Definiciones de descargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Obtiene la definición de una plantilla para descargas de datos.",
     *     description="Este endpoint permite obtener la definición de una plantilla existente por su ID.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición de plantilla.",
     *         @OA\Schema(
     *             type="integer",
     *             example=18
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="data", type="object",
     *                     @OA\Property(property="id", type="integer", example=18),
     *                     @OA\Property(property="nombre", type="string", example="roles"),
     *                     @OA\Property(property="referencia_nombre", type="string", example="roles"),
     *                     @OA\Property(property="no_fila_encabezados", type="integer", example=5),
     *                     @OA\Property(property="descripcion", type="string", example="The roles edited"),
     *                     @OA\Property(property="mapa_estructura", type="object",
     *                         @OA\Property(property="hojas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="nombre", type="string", example="roles"),
     *                                 @OA\Property(property="posicion_hoja", type="integer", example=0),
     *                                 @OA\Property(property="columnas", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="tipo", type="string", example="alfábetico"),
     *                                         @OA\Property(property="nombre", type="string", example="Rol"),
     *                                         @OA\Property(property="posicion", type="integer", example=0),
     *                                         @OA\Property(property="requerido", type="boolean", example=true),
     *                                         @OA\Property(property="registro_unico", type="boolean", example=true)
     *                                     )
     *                                 ),
     *                                 @OA\Property(property="campos_tabla", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="campo", type="string", example="name"),
     *                                         @OA\Property(property="tabla", type="string", example="roles")
     *                                     )
     *                                 ),
     *                                 @OA\Property(property="campos_relaciones", type="array", @OA\Items(type="object"))
     *                             )
     *                         )
     *                     ),
     *                     @OA\Property(property="mapa_modelo", type="object",
     *                         @OA\Property(property="hojas", type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="tabla", type="string", example="roles"),
     *                                 @OA\Property(property="nombre", type="string", example="roles"),
     *                                 @OA\Property(property="columnas", type="array",
     *                                     @OA\Items(
     *                                         type="object",
     *                                         @OA\Property(property="columna_bd", type="string", example="name"),
     *                                         @OA\Property(property="columna_archivo", type="integer", example=0)
     *                                     )
     *                                 )
     *                             )
     *                         )
     *                     ),
     *                     @OA\Property(property="tipo_archivo", type="string", example="excel"),
     *                     @OA\Property(property="created_at", type="string", format="date-time", example="2025-01-01T23:46:39.000000Z"),
     *                     @OA\Property(property="updated_at", type="string", format="date-time", example="2025-01-02T00:11:24.000000Z")
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición de plantilla",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=404),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="No se encontró la definición de la plantilla de descarga.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="Ocurrió un error al obtener la definición del archivo.")
     *             )
     *         )
     *     )
     * )
     * @throws AuthorizationException
     */

    public function show(int $id): JsonResponse
    {
        if (!$this->hasPermission('visualizar-definiciones-exportacion')) {
            throw new AuthorizationException();
        }
        $fileDefinition = ILCPlantillaDescarga::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición de la plantilla de descarga.');
        }

        try {
            $modelMap = is_string($fileDefinition->mapa_modelo)
                ? json_decode($fileDefinition->mapa_modelo, true)
                : $fileDefinition->mapa_modelo;

            if (!is_array($modelMap)) {
                throw new Exception('Formato inválido en el campo mapa_modelo');
            }

            $processedTables = [];
            foreach ($modelMap['tablas'] as $tabla) {

                $databaseFields = Schema::getColumnListing($tabla['tabla']);
                $savedFields = collect($tabla['campos'])->keyBy('nombre');

                $mergedFields = collect($databaseFields)->map(function ($columnName) use ($savedFields, $tabla) {
                    $savedField = $savedFields->get($columnName);
                    return [
                        'nombre' => $columnName,
                        'apodo' => $savedField['apodo'] ?? $columnName,
                        'tipo' => $this->mapFieldType($this->getColumnType($tabla['tabla'], $columnName)),
                        'tabla' => $tabla['tabla'],
                        'requerido' => $savedField['requerido'] ?? false,
                        'forTemplate' => $savedField['forTemplate'] ?? false,
                    ];
                });

                if ($tabla['tabla'] === $modelMap['tablas'][0]['tabla']) {
                    $processedTables['campos'] = $mergedFields;
                } else {
                    if (!isset($processedTables['relaciones'])) {
                        $processedTables['relaciones'] = [];
                    }
                    $processedTables['relaciones'][] = [
                        'tabla' => $tabla['tabla'],
                        'campos' => $mergedFields
                    ];
                }
            }

            $response = [
                    'id' => $fileDefinition->id,
                    'nombre' => $fileDefinition->nombre,
                    'referencia_nombre' => $fileDefinition->referencia_nombre,
                    'descripcion' => $fileDefinition->descripcion,
                    'mapa_modelo' => $modelMap,
                    'created_at' => $fileDefinition->created_at,
                    'updated_at' => $fileDefinition->updated_at,
                ] + $processedTables;

            return $this->sendResponse(['data' => $response]);
        } catch (Exception $e) {
            Log::error('Error al obtener la definición del archivo: ' . $e->getMessage());
            return $this->sendError('Ocurrió un error al obtener la definición del archivo.', 500);
        }
    }

    /**
     * @OA\Put(
     *     path="/api/data-exporter/{id}",
     *     operationId="updateDataExporter",
     *     tags={"Definiciones de descargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Actualiza la definición de una plantilla para descargas de datos.",
     *     description="Este endpoint permite actualizar una plantilla existente por su ID. Verifica la existencia de las tablas y campos antes de actualizar la definición.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición de plantilla.",
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         description="Datos de la plantilla a ser actualizados.",
     *         @OA\JsonContent(
     *             type="object",
     *             required={"nombre", "referencia_nombre", "descripcion", "tipo_archivo", "mapa_estructura", "mapa_modelo"},
     *             @OA\Property(property="nombre", type="string", example="roles"),
     *             @OA\Property(property="referencia_nombre", type="string", example="roles"),
     *             @OA\Property(property="no_fila_encabezados", type="integer", example=5),
     *             @OA\Property(property="descripcion", type="string", example="Los roles"),
     *             @OA\Property(property="tipo_archivo", type="string", example="excel"),
     *             @OA\Property(
     *                 property="mapa_estructura",
     *                 type="object",
     *                 @OA\Property(
     *                     property="hojas",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="nombre", type="string", example="roles"),
     *                         @OA\Property(property="posicion_hoja", type="integer", example=0),
     *                         @OA\Property(
     *                             property="columnas",
     *                             type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="posicion", type="integer", example=0),
     *                                 @OA\Property(property="nombre", type="string", example="Rol"),
     *                                 @OA\Property(property="tipo", type="string", example="alfábetico"),
     *                                 @OA\Property(property="requerido", type="boolean", example=true),
     *                                 @OA\Property(property="registro_unico", type="boolean", example=true)
     *                             )
     *                         )
     *                     )
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="mapa_modelo",
     *                 type="object",
     *                 @OA\Property(
     *                     property="hojas",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="nombre", type="string", example="roles"),
     *                         @OA\Property(property="tabla", type="string", example="roles"),
     *                         @OA\Property(
     *                             property="columnas",
     *                             type="array",
     *                             @OA\Items(
     *                                 type="object",
     *                                 @OA\Property(property="columna_archivo", type="integer", example=0),
     *                                 @OA\Property(property="columna_bd", type="string", example="name")
     *                             )
     *                         )
     *                     )
     *                 )
     *             ),
     *             @OA\Property(property="created_at", type="string", format="date-time", example=null),
     *             @OA\Property(property="updated_at", type="string", format="date-time", example=null)
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="Definición de archivo actualizada exitosamente.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición de plantilla",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=404),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="No se encontró la definición de la plantilla de descarga.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Error de validación, tabla o campo no existe",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=422),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="La tabla roles no existe en la base de datos.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="Ocurrió un error al actualizar la definición de la plantilla.")
     *             )
     *         )
     *     )
     * )
     * @throws AuthorizationException
     */

    public function update(TemplateDefinitionRequest $request, int $id): JsonResponse
    {
        if (!$this->hasPermission('editar-definiciones-exportacion')) {
            throw new AuthorizationException();
        }
        $fileDefinition = ILCPlantillaDescarga::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición de la plantilla de descarga.');
        }

        try {

            foreach ($request->mapa_modelo['tablas'] as $tabla) {
                if (!Schema::hasTable($tabla['tabla'])) {
                    return $this->sendError("La tabla {$tabla['tabla']} no existe en la base de datos.", 422);
                }

                $tableColumns = Schema::getColumnListing($tabla['tabla']);
                foreach ($tabla['campos'] as $campo) {
                    if (!in_array($campo['nombre'], $tableColumns)) {
                        return $this->sendError("El campo {$campo['nombre']} no existe en la tabla {$tabla['tabla']}.", 422);
                    }
                }
            }

            $fileDefinition->update([
                'nombre' => $request->nombre,
                'referencia_nombre' => $request->nombre,
                'descripcion' => $request->descripcion,
                'mapa_modelo' => $request->mapa_modelo,
            ]);

            return $this->sendResponse([
                'message' => 'Definición de plantilla actualizada exitosamente',
                'data' => $fileDefinition
            ]);
        } catch (Exception $e) {
            Log::error('Error al actualizar la definición de la plantilla: ' . $e->getMessage());
            return $this->sendError('Ocurrió un error al actualizar la definición de la plantilla.', 500);
        }
    }


    /**
     * @OA\Delete(
     *     path="/api/data-exporter/{id}",
     *     operationId="destroyDataExporter",
     *     tags={"Definiciones de descargas"},
     *     security={{"BearerAuth":{}}},
     *     summary="Elimina la definición de una plantilla para descargas de datos.",
     *     description="Este endpoint permite eliminar una plantilla existente por su ID.",
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="ID de la definición de plantilla a eliminar.",
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Operación exitosa",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="result", type="string", example="La definición de la plantilla de descarga se ha eliminado exitosamente")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="No se encontró la definición de plantilla",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=404),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="No se encontró la definición de la descarga solicitada.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno del servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(
     *                 property="result",
     *                 type="object",
     *                 @OA\Property(property="message", type="string", example="Ha ocurrido un error al intentar eliminar la definición de la plantilla.")
     *             )
     *         )
     *     )
     * )
     * @throws AuthorizationException
     */

    public function destroy(int $id): JsonResponse
    {
        if (!$this->hasPermission('eliminar-definiciones-exportacion')) {
            throw new AuthorizationException();
        }
        $fileDefinition = ILCPlantillaDescarga::find($id);

        if (!$fileDefinition) {
            return $this->sendError('No se encontró la definición de la descarga solicitada.');
        }

        try {
            $fileDefinition->delete();
            return $this->sendResponse('La definición de la plantilla de descarga se ha eliminado exitosamente');
        } catch (Exception $e) {
            Log::error('Error al eliminar la definición de la plantilla: ' . $e->getMessage());

            return $this->sendError('Ha ocurrido un error al intentar eliminar la definición de la plantilla.');
        }
    }


    /**
     * Mapea el tipo de campo
     *
     * @param string $databaseType
     * @return string
     */
    private function mapFieldType(string $databaseType): string
    {
        $typeMap = [
            'int' => 'numerico',
            'bigint' => 'numerico',
            'decimal' => 'numerico',
            'float' => 'numerico',
            'double' => 'numerico',
            'varchar' => 'alfabetico',
            'char' => 'alfabetico',
            'text' => 'alfabetico',
            'date' => 'fecha',
            'datetime' => 'fecha',
            'timestamp' => 'fecha',
            'boolean' => 'verdadero/falso',
        ];

        foreach ($typeMap as $dbType => $mappedType) {
            if (str_contains(strtolower($databaseType), $dbType)) {
                return $mappedType;
            }
        }

        return 'alfabetico';
    }


    /**
     * Obtiene el tipo de columna de un campo
     *
     * @param string $table
     * @param string $column
     * @return string
     */
    private function getColumnType(string $table, string $column): string
    {
        return Schema::getColumnType($table, $column);
    }



    /**
     * Obtiene los campos de una tabla y en caso de aplicar, incluye
     * también los datos de otras tablas relacionadas
     *
     * @param string $table
     * @return JsonResponse
     */
    public function getTableFieldsWithRelationships(string $table): JsonResponse
    {
        if (!$this->hasPermission('guardar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }

        try {
            $fields = $this->getTableFieldsWithDataType($table);
            $relationships = $this->getRelatedTablesWithFields($table, true);

            $response = [
                'campos' => $fields,
                'relaciones' => $relationships,
            ];

            return $this->sendResponse([ 'data' => $response ]);

        } catch (Exception $e) {
            return $this->sendError(
                'Se produjo un error al procesar al tratar de obtener los campos con relaciones: ' . $e->getMessage(),
                500
            );
        }
    }


    /**
     * Retorna las tablas de la base de datos
     *
     * @return JsonResponse
     */
    public function getTablesInDatabase(Request $request): JsonResponse
    {
        if (!$this->hasPermission('guardar-definiciones-exportacion') || !$this->hasPermission('editar-definiciones-exportacion') ) {
            throw new AuthorizationException();
        }

        try {
            $tableNames= $this->getAllTablesInSchema();

            return $this->sendResponse([
                'data' => $tableNames
            ]);

        } catch (Exception $e) {

            return $this->sendError(
                'Error obteniendo las tablas: ' . $e->getMessage(),
                500
            );
        }
    }


    /**
     *  Genera la plantilla de excel en base a la tabla solicitada
     *  Puede incluir datos de varias tablas, según se haya especificado en la definición de la plantilla
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function exportToExcel(Request $request)
    {
        // Autorización.
        if (!$this->hasPermission('exportar-mediante-template')) {
            throw new AuthorizationException();
        }

        $slug = $request->get('slug');

        if (empty($slug)) {
            return $this->sendError(['error' => 'No se especificó la tabla.'], 400);
        }

        $definition = ILCPlantillaDescarga::where('referencia_nombre', '=', $slug)->first();

        if (!$definition) {
            return $this->sendError(['error' => 'No se encontró la definición de la exportación.'], 404);
        }

        $modelMap = $definition->mapa_modelo;

        if (!isset($modelMap['tablas']) || empty($modelMap['tablas'])) {
            return $this->sendError(['error' => 'La definición de la exportación es inválida, no está definida la tabla.'], 400);
        }

        $nameOfFile = $request->query('fileName', $definition->nombre);

        return $this->exporterService->generateMultiTableExcel(
            $request,
            $modelMap,
            $nameOfFile
        );
    }
}
