<?php

namespace ILC\CargaArchivos\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Support\Facades\Log;
use Validator;

/**
 * @OA\Info(
 *      version="1.0.0",
 *      x={
 *          "logo": {
 *              "url": "https://avatars.githubusercontent.com/u/165195165?s=200&v=4"
 *          }
 *      },
 *      title="API paquete para definiciones de cargas y exportaciones de datos",
 *      description="Archivo de documentación de API del paquete para Definiciones de Cargas y Exportaciones de datos con OpenAPI",
 *      @OA\Contact(
 *          email="fernando@neoideas.mx"
 *      ),
 *      @OA\License(
 *         name="Apache 2.0",
 *         url="https://www.apache.org/licenses/LICENSE-2.0.html"
 *      ),
 *      @OA\SecurityScheme(
 *          securityScheme="BearerAuth",
 *          type="http",
 *          scheme="bearer"
 *      ),
 * )
 */
class Controller extends BaseController
{


    /**
     * Retorna el resultado de una petición exitosa.
     *
     * @param $result
     * @param int $code
     * @return JsonResponse
     */
    protected function sendResponse($result = null, int $code = 200): JsonResponse
    {
        $response = array('success' => true, 'code' => $code);
        if (!$result == null){
            if ($result instanceof LengthAwarePaginator ||
                $result instanceof Paginator) {

                $response['result'] = [
                    'data' => $result->items(),
                    'pagination' => [
                        'total' => $result->total(),
                        'count' => $result->count(),
                        'per_page' => $result->perPage(),
                        'current_page' => $result->currentPage(),
                        'total_pages' => $result->lastPage(),
                    ],
                ];
            } else {

                $response['result'] = $result;
            }
        }

        return response()->json($response, $code);
    }


    /**
     * return el error de una petición.
     *
     * @param $error
     * @param int $code
     * @return JsonResponse
     */
    protected function sendError($error, int $code = 404): JsonResponse
    {
        $code = ($code == 0 ? 500 : $code);
        $response = [
            'success' => false,
            'code' => $code,
            'result' => array('message' => ($code == 401 ? 'Acceso No Autorizado' : $error))
        ];

        if($code == 500){ Log::error('error', $response); }

        return response()->json($response, $code);
    }


    /**
     * Revisa si el usuario está autenticado y tiene permiso de realizar la acción.
     *
     * @param $permission
     * @return bool
     */
    protected function hasPermission($permission): bool
    {
        return auth()->check() && auth()->user()->hasPermissionTo($permission);
    }


}
