<?php

namespace ILC\CargaArchivos\Imports;

use Exception;
use ILC\CargaArchivos\Traits\DatabaseMappable;
use ILC\CargaArchivos\Traits\HandlesImportedDataValidation;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use ILC\CargaArchivos\Traits\ProgressTrackable;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Rap2hpoutre\FastExcel\FastExcel;
use App\Models\Role;
use App\Models\User;
use ILC\AdminUsuarios\Models\Perfil;
use Illuminate\Http\JsonResponse;
use ILC\CargaArchivos\Events\ModelSaved;

class DataImport
{
    use ProgressTrackable, HandlesImportedDataValidation, DatabaseMappable, HandlesRelatedTables;

    protected array $importDetails = [
        'file_name' => '',
        'total_records_in_file' => 0,
        'records_added' => 0,
        'records_updated' => 0,
        'records_with_errors' => 0,
        'errors_detail' => [],
    ];

    protected array $structureMap;
    protected array $modelMap;
    protected string $tableName;
    protected array $validatedData = [];
    protected array $failedData = [];
    protected int $successfulRegistrations = 0;
    protected int $failedRegistrations = 0;
    protected UploadedFile $file;
    protected int $headersLineNumber;
    protected int $currentRowIndex;
    protected string $currentSheetName;
    protected int $currentSheetIndex = 0;
    protected $storagePath;

    /**
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @param array $structureMap
     * @param array $modelMap
     * @param string $tableName
     */
    public function __construct(UploadedFile $file, int $headersLineNumber, array $structureMap, array $modelMap, string $tableName)
    {
        $this->file = $file;
        $this->headersLineNumber = $headersLineNumber;
        $this->structureMap = $structureMap;
        $this->modelMap = $modelMap;
        $this->tableName = $tableName;
        $this->currentRowIndex = 0;
        $this->currentSheetName = '';
        $this->importDetails['file_name'] = $file->getClientOriginalName();
        $this->storagePath = $this->storeFile($file);
    }

    /**
     * Guarda temporalmente el archivo para procesarlo
     *
     * @param UploadedFile $file
     * @return string
     */
    protected function storeFile(UploadedFile $file): string
    {
        $path = 'imports/' . uniqid() . '.' . $file->getClientOriginalExtension();
        Storage::put($path, file_get_contents($file->getRealPath()));
        return $path;
    }

    /**
     * Procesa una fila a la vez y la guarda en la BD
     *
     * @param array $row
     * @param string $tableName
     * @param int $rowIndex
     * @return array|null
     */
    protected function processRow(array $row, string $tableName, int $rowIndex)
    {
        try {
            $this->currentRowIndex = $rowIndex;
            $modelClass = $this->getModelClassForTable($tableName);

            if (!$modelClass) {
                throw new Exception("No se encontró el modelo para la tabla {$tableName}");
            }

            $sheetIndex = $this->currentSheetIndex;
            $sheetStructure = $this->structureMap['hojas'][$sheetIndex] ?? null;

            if ($sheetStructure) {
                try {
                    $mappedRow = $this->mapRowToDatabase(
                        $row,
                        $rowIndex,
                        $sheetIndex,
                        $sheetStructure,
                        $this->modelMap['hojas'][$sheetIndex] ?? []
                    );

                    if (!$mappedRow) {
                        return null;
                    }
                } catch (Exception $e) {
                    throw new Exception("Error de validación: " . $e->getMessage());
                }
            } else {
                $mappedRow = $row;
            }

            $uniqueFields = $this->getUniqueFieldForDatabaseTable($tableName, $mappedRow, $this->currentSheetIndex);
            $mappingResult = $this->processFieldMapping($mappedRow, $tableName);

            if (!empty($mappingResult['errors'])) {
                throw new Exception("Errores de validación: " . implode(', ', $mappingResult['errors']));
            }

            $row = $mappingResult['data'];
            $relatedModel = $row['modelo_relacionado'] ?? null;
            unset($row['modelo_relacionado'], $row['tabla_relacionada']);

            if (!empty($row)) {
                $row['updated_at'] = now();
                $modelInstance = new $modelClass;

                if (!empty($uniqueFields)) {
                    $exists = $modelInstance->newQuery()
                        ->where($uniqueFields)
                        ->exists();

                    if ($exists) {
                        $modelInstance->newQuery()
                            ->where($uniqueFields)
                            ->update($row);

                        $createdModel = $modelInstance->newQuery()
                            ->where($uniqueFields)
                            ->first();

                        $this->incrementUpdatedRows();
                    } else {
                        $createdModel = $modelInstance->create(array_merge($uniqueFields, $row));
                        $this->incrementAddedRows();
                    }
                } else {
                    $createdModel = $modelInstance->create($row);
                    $this->incrementAddedRows();
                }

                if ($relatedModel && $createdModel) {
                    $relationMethod = Str::camel(class_basename(get_class($relatedModel)));

                    if (method_exists($createdModel, $relationMethod)) {
                        $createdModel->$relationMethod()->associate($relatedModel);
                        $createdModel->save();
                    } else {
                        Log::warning("No se encontró la relación para el modelo relacionado '{$relationMethod}'.");
                    }
                }

                if ($createdModel) {
                    event(new ModelSaved($createdModel));
                }

                $this->updateUploadProgress(
                    $this->successfulRegistrations,
                    $this->failedRegistrations,
                    $this->importDetails['total_records_in_file']
                );

                $this->successfulRegistrations++;

                return $row;
            }
        } catch (Exception $e) {
            $this->importDetails['records_with_errors']++;
            $this->failedRegistrations++;
            $errors = [$e->getMessage()];

            $this->failedData[] = [
                'hoja' => $this->currentSheetName,
                'fila' => $this->currentRowIndex,
                'errors' => $errors
            ];

            Log::error("Error al procesar la fila {$rowIndex} en la hoja {$this->currentSheetName}: " . $e->getMessage());
        }

        return null;
    }

    protected function getModelClassForTable($tableName): ?string
    {
        $models = config('cargaarchivos.models');

        return $models[$tableName] ?? null;
    }

    /**
     * @throws Exception
     */
    private function processFieldMapping(array $row, string $tableName): array
    {
        $errors = [];
        $preparedData = [];
        $columns = Schema::getColumnListing($tableName);
        $fieldsFromRelatedTables = $this->getRelatedTablesWithFields($tableName);

        foreach ($row as $fileField => $value) {
            if (in_array($fileField, $columns)) {
                $preparedData[$fileField] = $value;
            } else {
                $relatedTable = collect($fieldsFromRelatedTables)
                    ->first(fn($table) => in_array($fileField, $table['campos']));

                if ($relatedTable) {
                    $relatedTableName = $relatedTable['tabla'];

                    $modelClass = $this->getModelClassForTable($relatedTableName);

                    if ($modelClass) {
                        $relatedRecord = $modelClass::where($fileField, $value)->first();
                        if ($relatedRecord) {
                            $preparedData['modelo_relacionado'] = $relatedRecord;
                            $preparedData['tabla_relacionada'] = $tableName;
                        } else {
                            $errors[] = "El valor '$value' para el campo '$fileField' no se encuentra en la tabla '$relatedTableName'.";
                        }
                    } else {
                        $errors[] = "No se encontró un modelo para la tabla '$relatedTableName'.";
                    }
                }
            }
        }

        return [
            'errors' => $errors,
            'data' => $preparedData,
        ];
    }

    protected function incrementAddedRows(): void
    {
        $this->importDetails['records_added']++;
        Log::info(['registros_agregados', $this->importDetails['records_added']]);
    }

    protected function incrementUpdatedRows(): void
    {
        $this->importDetails['records_updated']++;
        Log::info(['registros_actualizados', $this->importDetails['records_updated']]);
    }

    public function getImportDetails(): array
    {
        return $this->importDetails;
    }

    public function getImportProgress(): JsonResponse
    {
        $progress = $this->getUploadProgress();
        Log::info('avance: ', [$progress]);
        return response()->json($progress);
    }

    /**
     * Obtiene los datos desde el archivo Excel usando FastExcel
     *
     * @return array
     */
    public function get(): array
    {
        try {
            $filePath = Storage::path($this->storagePath);
            $fastExcel = new FastExcel;

            $fileExtension = pathinfo($filePath, PATHINFO_EXTENSION);
            $isCSV = strtolower($fileExtension) === 'csv';

            $this->importDetails['total_records_in_file'] = 0;
            $processedData = [];

            if ($isCSV) {
                $sheetMapping = $this->modelMap['hojas'][0] ?? null;
                if (!$sheetMapping) {
                    throw new Exception("No se encontraron definiciones para la hoja CSV.");
                }

                $this->currentSheetName = $sheetMapping['nombre'];
                $this->currentSheetIndex = 0;
                $rowCount = 0;
                $rowsToSkip = $this->headersLineNumber;
                $fastExcel->configureCsv(',', '"', '\\', $rowsToSkip);

                foreach ($fastExcel->import($filePath) as $rowIndex => $row) {
                    $rowCount++;
                    $mappedRow = $this->mapRowToDatabase(
                        $row,
                        $rowCount,
                        0,
                        $this->structureMap['hojas'][0],
                        $sheetMapping
                    );

                    if (!empty($mappedRow)) {
                        $this->validatedData[] = $mappedRow;
                        $processedData[$sheetMapping['tabla']][] = $mappedRow;
                        $this->successfulRegistrations++;
                    }
                }

                $this->importDetails['total_records_in_file'] = $rowCount;
                $this->importDetails['sheets'][0]['total_records'] = $rowCount;
                $this->importDetails['sheets'][0]['name'] = $sheetMapping['nombre'];

            } else {
                $sheetCount = count($this->modelMap['hojas'] ?? []);

                for ($sheetIndex = 0; $sheetIndex < $sheetCount; $sheetIndex++) {
                    $sheetMapping = $this->modelMap['hojas'][$sheetIndex] ?? null;
                    $sheetStructure = $this->structureMap['hojas'][$sheetIndex] ?? null;

                    if (!$sheetMapping || !$sheetStructure) {
                        Log::warning("No se encontraron definiciones para la hoja con índice {$sheetIndex}.");
                        continue;
                    }

                    $this->currentSheetName = $sheetMapping['nombre'];
                    $this->currentSheetIndex = $sheetIndex;
                    $rowCount = 0;

                    $headerRow = $this->headersLineNumber - 1;
                    foreach ($fastExcel->sheet($sheetIndex + 1)->import($filePath, null, $headerRow) as $rowIndex => $row) {
                        $rowCount++;
                        $mappedRow = $this->mapRowToDatabase(
                            $row,
                            $rowCount,
                            $sheetIndex,
                            $sheetStructure,
                            $sheetMapping
                        );

                        if (!empty($mappedRow)) {
                            $this->validatedData[] = $mappedRow;
                            $processedData[$sheetMapping['tabla']][] = $mappedRow;
                            $this->successfulRegistrations++;
                        }
                    }

                    $this->importDetails['total_records_in_file'] += $rowCount;
                    $this->importDetails['sheets'][$sheetIndex]['total_records'] = $rowCount;
                    $this->importDetails['sheets'][$sheetIndex]['name'] = $sheetMapping['nombre'];
                }
            }

            return [
                'success' => true,
                'data' => $processedData,
                'errors' => $this->failedData,
            ];

        } catch (Exception $e) {
            Log::error('Error al importar los datos: ' . $e->getMessage());
            Log::error($e->getTraceAsString());

            return [
                'success' => false,
                'message' => 'Error al importar',
                'error' => $e->getMessage(),
            ];
        } finally {
            if (Storage::exists($this->storagePath)) {
                Storage::delete($this->storagePath);
            }
        }
    }

    /**
     * Obtiene la información en formato JSON
     *
     * @return string
     */
    public function getJson(): string
    {
        return json_encode($this->get());
    }

    /**
     * Guarda directamente en al BD usando chunking
     *
     * @return array
     */
    public function store(): array
    {
        try {
            $filePath = Storage::path($this->storagePath);
            $fastExcel = new FastExcel;
            $fileExtension = pathinfo($filePath, PATHINFO_EXTENSION);
            $isCSV = strtolower($fileExtension) === 'csv';

            $this->importDetails['total_records_in_file'] = 0;
            $this->importDetails['errors_detail'] = [];

            $chunkSize = 100;
            $rowsInCurrentChunk = 0;

            if ($isCSV) {
                $sheetMapping = $this->modelMap['hojas'][0] ?? null;

                if (!$sheetMapping) {
                    throw new Exception("No se encontraron definiciones para la hoja CSV.");
                }

                $this->currentSheetName = $sheetMapping['tabla'];
                $this->currentSheetIndex = 0;
                $rowCount = 0;
                $rowsToSkip = $this->headersLineNumber - 1;
                $fastExcel->configureCsv(',', '"', '\\', $rowsToSkip);

                foreach ($fastExcel->import($filePath, function($row) use (&$rowCount, &$rowsInCurrentChunk, $chunkSize) {
                    $rowCount++;

                    if ($rowsInCurrentChunk === 0) {
                        DB::beginTransaction();
                    }

                    $this->processRow($row, $this->currentSheetName, $rowCount);
                    $rowsInCurrentChunk++;

                    if ($rowsInCurrentChunk >= $chunkSize) {
                        DB::commit();
                        $rowsInCurrentChunk = 0;
                        gc_collect_cycles();
                    }

                    return null;
                }, 0) as $_) {
                }

                if ($rowsInCurrentChunk > 0) {
                    DB::commit();
                }

                $this->importDetails['total_records_in_file'] = $rowCount;

            } else {
                $sheetCount = count($this->modelMap['hojas'] ?? []);
                $totalRows = 0;

                for ($sheetIndex = 0; $sheetIndex < $sheetCount; $sheetIndex++) {
                    $sheetMapping = $this->modelMap['hojas'][$sheetIndex] ?? null;

                    if (!$sheetMapping) {
                        Log::warning("No se encontraron definiciones para la hoja con índice {$sheetIndex}.");
                        continue;
                    }

                    $this->currentSheetName = $sheetMapping['tabla'];
                    $this->currentSheetIndex = $sheetIndex;
                    $rowCount = 0;
                    $rowsInCurrentChunk = 0;

                    $headerRow = $this->headersLineNumber - 1;
                    foreach ($fastExcel->sheet($sheetIndex + 1)->import($filePath, function($row) use (&$rowCount, &$rowsInCurrentChunk, $chunkSize) {
                        $rowCount++;

                        if ($rowsInCurrentChunk === 0) {
                            DB::beginTransaction();
                        }

                        $this->processRow($row, $this->currentSheetName, $rowCount);
                        $rowsInCurrentChunk++;

                        if ($rowsInCurrentChunk >= $chunkSize) {
                            DB::commit();
                            $rowsInCurrentChunk = 0;
                            gc_collect_cycles();
                        }

                        return null;
                    }, $headerRow) as $_) {
                    }

                    if ($rowsInCurrentChunk > 0) {
                        DB::commit();
                    }

                    $totalRows += $rowCount;
                    $this->importDetails['sheets'][$sheetIndex]['total_records'] = $rowCount;
                }

                $this->importDetails['total_records_in_file'] = $totalRows;
            }

            Log::info('Registros guardados correctamente.');

            return [
                'success' => true,
                'importDetails' => [
                    'file_name' => $this->importDetails['file_name'],
                    'total_records' => $this->importDetails['total_records_in_file'],
                    'records_added' => $this->importDetails['records_added'],
                    'records_updated' => $this->importDetails['records_updated'],
                    'records_with_errors' => count($this->failedData),
                    'errors_detail' => $this->failedData ?? [],
                ],
                'message' => count($this->failedData) == 0
                    ? 'Los registros se han guardado correctamente.'
                    : 'Hubo ' . count($this->failedData) . ' errores durante la importación.',
            ];

        } catch (Exception $e) {
            if (DB::transactionLevel() > 0) {
                DB::rollBack();
            }

            Log::error('Error al guardar los registros: ' . $e->getMessage());
            Log::error($e->getTraceAsString());

            return [
                'success' => false,
                'errors' => $this->failedData,
                'message' => 'Error al guardar los registros: ' . $e->getMessage(),
            ];

        } finally {
            if (Storage::exists($this->storagePath)) {
                Storage::delete($this->storagePath);
            }
        }
    }

    private function getUniqueFieldForDatabaseTable(string $tableName, array $rowData, int $index): array
    {
        $uniqueField = [];
        $columnMapping = $this->structureMap['hojas'][$index]['columnas'] ?? [];

        foreach ($columnMapping as $column) {
            if (isset($column['registro_unico']) && $column['registro_unico'] === true) {
                foreach ($this->modelMap['hojas'][$index]['columnas'] as $modelColumn) {
                    if ($column['posicion'] === $modelColumn['columna_archivo'] && isset($modelColumn['columna_bd'])) {
                        $uniqueField[$modelColumn['columna_bd']] = $rowData[$modelColumn['columna_bd']] ?? null;
                    }
                }
            }
        }

        return $uniqueField;
    }
}
