<?php

namespace ILC\CargaArchivos\Http\Controllers;

use ILC\CargaArchivos\Exceptions\AuthorizationException;
use ILC\CargaArchivos\Http\Requests\DynamicCatalogRequest;
use ILC\CargaArchivos\Traits\CatalogsQueryTrait;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class CatalogController extends Controller
{
    use HandlesRelatedTables, CatalogsQueryTrait;

    /**
     * Muestra los registros de un catálogo
     *
     * @param string $table
     * @return JsonResponse
     */
    public function index(string $table, Request $request): JsonResponse
    {
        if (!$this->hasPermission('visualizar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }

        $validatedCatalogInfo = $this->validateCatalogWithConfigOptions($table);

        if (isset($validatedCatalogInfo['error'])) {
            return $this->sendError(
                $validatedCatalogInfo['error'],
                $validatedCatalogInfo['status'] ?? 400
            );
        }

        $modelClass = $validatedCatalogInfo['modelClass'];
        $filteredColumns = $validatedCatalogInfo['filteredColumns'];
        $recordsQuery = $modelClass::select($filteredColumns);

        if ($request->filled('search')) {
            $search = $request->search;

            $searchableFields = config('cargaarchivos.searchable_fields.'.$table, []);

            if (!empty($searchableFields)) {
                $recordsQuery->where(function ($query) use ($search, $searchableFields) {
                    foreach ($searchableFields as $field) {
                        $query->orWhere($field, 'ilike', "%$search%");
                    }
                });
            }
        }

        $this->applySorting($recordsQuery, $request);
        $perPage = $request->get('perPage', 10);
        $catalogRecords = $recordsQuery->paginate($perPage);

        return $this->sendResponse([
            'success' => true,
            'total' => $catalogRecords->total(),
            'filter' => $catalogRecords->total(),
            'pages' => $catalogRecords->lastPage(),
            'data' => $catalogRecords->items(),
        ]);
    }

    public function getTableHeaders(string $table): JsonResponse
    {
        $catalogFieldsMapping = config('cargaarchivos.catalog_fields_mapping');

        if (!isset($catalogFieldsMapping[$table])) {
            throw new \InvalidArgumentException("No se encontró la tabla '{$table}' en el archivo de configuración.");
        }

        $fields = $catalogFieldsMapping[$table];
        $headers = [];

        foreach ($fields as $key => $title) {
            $headers[] = [
                'title' => ucfirst($title),
                'key' => $key
            ];
        }

        $headers[] = [
            'title' => 'Acciones',
            'key' => 'actions',
            'sortable' => false,
            'align' => 'end',
        ];

        return $this->sendResponse([
            'data' => $headers
        ]);
    }


    /**
     * @OA\Post(
     *     path="/api/catalogs/{table}",
     *     operationId="storeCatalog",
     *     tags={"Catálogos Dinámicos"},
     *     summary="Almacena un registro en un catálogo dinámico.",
     *     description="Almacena un nuevo registro desde un catálogo dinámico en la tabla especificada.",
     *     @OA\Parameter(
     *         name="table",
     *         in="path",
     *         required=true,
     *         description="El nombre de la tabla donde se almacenará el registro",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         description="Los datos que se almacenarán en la tabla del catálogo dinámico",
     *         @OA\JsonContent(
     *             type="object",
     *             example={"name": "ver-algun-listado", "guard_name": "web"}
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Registro creado correctamente",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="data", type="string", example="Se ha guardado el registro correctamente.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Error de validación o nombre de tabla incorrecto",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=400),
     *             @OA\Property(property="error", type="string", example="Mensaje de error describiendo el error.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error interno de servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="error", type="string", example="Error al intentar crear el registro: [detalle del error]")
     *         )
     *     )
     * )
     */

    public function store(DynamicCatalogRequest $request, string $table): JsonResponse
    {
        if (!$this->hasPermission('guardar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }
        $validatedCatalogInfo = $this->validateCatalogWithConfigOptions($table);

        if (isset($validatedCatalogInfo['error'])) {
            return $this->sendError(
                $validatedCatalogInfo['error'],
                $validatedCatalogInfo['status'] ?? 400
            );
        }

        $modelClass = $validatedCatalogInfo['modelClass'];

        try {
            $validatedData = $request->validated();
            $modelClass::create($validatedData);

            return $this->sendResponse(['data' => 'Se ha guardado el registro correctamente.']);

        } catch (\Exception $e) {
            return $this->sendError('Error al intentar crear el registro: ' . $e->getMessage(), 500);
        }
    }

    public function show(string $table, $id, Request $request): JsonResponse
    {
        if (!$this->hasPermission('visualizar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }
        $validatedCatalogInfo = $this->validateCatalogWithConfigOptions($table);

        if (isset($validatedCatalogInfo['error'])) {
            return $this->sendError(
                $validatedCatalogInfo['error'],
                $validatedCatalogInfo['status'] ?? 400
            );
        }

        $modelClass = $validatedCatalogInfo['modelClass'];
        $filteredColumns = $validatedCatalogInfo['filteredColumns'];
        $record = $modelClass::select($filteredColumns)->find($id);

        if (!$record) {
            return $this->sendError("El registro con la id {$id} no existe en la tabla '{$table}'.", 404);
        }

        return $this->sendResponse([
            'data' => $record
        ]);
    }


    /**
     * @OA\Put(
     *     path="/api/catalogs/{table}/{id}",
     *     operationId="updateCatalog",
     *     tags={"Catálogos Dinámicos"},
     *     summary="Actualiza un registro de un catálogo dinámico",
     *     description="Actualiza un registro de un catálogo dinámico.",
     *     @OA\Parameter(
     *         name="table",
     *         in="path",
     *         required=true,
     *         description="El nombre de la tabla donde se encuentra almancenado el registro",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="La ID del registro que se va a actualizar",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         description="Los datos a actualizar en el registro del catálogo dinámico",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(
     *                 property="name",
     *                 type="string",
     *                 description="El nombre del permiso",
     *                 example="permiso-de-ejemplo",
     *                 maxLength=255
     *             ),
     *             @OA\Property(
     *                 property="guard_name",
     *                 type="string",
     *                 description="El guard del permiso",
     *                 example="web",
     *                 maxLength=255
     *             ),
     *             required={"name", "guard_name"}
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Registro actualizado correctamente.",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="code", type="integer", example=200),
     *             @OA\Property(property="result", type="object",
     *                 @OA\Property(property="data", type="string", example="Se ha actualizado correctamente el registro.")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Error de validación o nombre de tabla inválido",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=400),
     *             @OA\Property(property="error", type="string", example="Mensaje describiendo el error")
     *         )
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Registro no encontrado",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=404),
     *             @OA\Property(property="error", type="string", example="El registro con la id {id} no existe en la tabla '{table}'.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error de servidor",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="code", type="integer", example=500),
     *             @OA\Property(property="error", type="string", example="Error al intentar actualizar el registro: [detalle del error]")
     *         )
     *     )
     * )
     */
    public function update(DynamicCatalogRequest $request, string $table, string $id): JsonResponse
    {
        if (!$this->hasPermission('editar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }
        $validatedCatalogInfo = $this->validateCatalogWithConfigOptions($table);

        if (isset($validatedCatalogInfo['error'])) {
            return $this->sendError(
                $validatedCatalogInfo['error'],
                $validatedCatalogInfo['status'] ?? 400
            );
        }

        $modelClass = $validatedCatalogInfo['modelClass'];
        $record = $modelClass::find($id);

        if (!$record) {
            return $this->sendError("El registro con la id {$id} no existe en la tabla '{$table}'.", 404);
        }

        try {
            $validatedData = $request->validated();

            if (isset($validatedData['password'])) {
                $validatedData['password'] = Hash::make($validatedData['password']);
            }

            $record->update($validatedData);

            return $this->sendResponse([
                'data' => 'Se ha actualizado correctamente el registro.'
            ]);
        } catch (\Exception $e) {
            return $this->sendError('Error al intentar actualizar el registro: ' . $e->getMessage(), 500);
        }
    }

    /**
     * @OA\Delete(
     *     path="/api/catalogs/{table}/{id}",
     *     operationId="deleteCatalog",
     *     tags={"Catálogos Dinámicos"},
     *     summary="Elimina un registro de un catálogo dinámico",
     *     description="Elimina un registro de un catálogo dinámico.",
     *     @OA\Parameter(
     *         name="table",
     *         in="path",
     *         required=true,
     *         description="El nombre de la tabla donde se encuentra almancenado el registro",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="La ID del registro que se va a actualizar",
     *         @OA\Schema(type="string")
     *      ),
     *      @OA\Response(
     *          response=200,
     *          description="Operación exitosa",
     *          @OA\JsonContent(
     *              type="object",
     *              @OA\Property(property="success", type="boolean", example=true),
     *              @OA\Property(property="code", type="integer", example=200),
     *              @OA\Property(property="result", type="string", example="La definición de la plantilla de descarga se ha eliminado exitosamente")
     *          )
     *      ),
     *      @OA\Response(
     *          response=404,
     *          description="No se encontró la definición de plantilla",
     *          @OA\JsonContent(
     *              type="object",
     *              @OA\Property(property="success", type="boolean", example=false),
     *              @OA\Property(property="code", type="integer", example=404),
     *              @OA\Property(
     *                  property="result",
     *                  type="object",
     *                  @OA\Property(property="message", type="string", example="No se encontró la definición de la descarga solicitada.")
     *              )
     *          )
     *      ),
     *      @OA\Response(
     *          response=500,
     *          description="Error interno del servidor",
     *          @OA\JsonContent(
     *              type="object",
     *              @OA\Property(property="success", type="boolean", example=false),
     *              @OA\Property(property="code", type="integer", example=500),
     *              @OA\Property(
     *                  property="result",
     *                  type="object",
     *                  @OA\Property(property="message", type="string", example="Ha ocurrido un error al intentar eliminar la definición de la plantilla.")
     *              )
     *          )
     *      )
     *  )
     */
    public function destroy(string $table, $id): JsonResponse
    {
        if (!$this->hasPermission('eliminar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }
        $catalogInfo = $this->validateCatalogWithConfigOptions($table);

        if (isset($catalogInfo['error'])) {
            return $this->sendError(
                $catalogInfo['error'],
                $catalogInfo['status'] ?? 400
            );
        }

        $modelClass = $catalogInfo['modelClass'];
        $record = $modelClass::find($id);

        if (!$record) {
            return $this->sendError(
                'No se encontró el registro.',
                400
            );
        }

        $record->delete();

        return $this->sendResponse([
            'data' => 'Se eliminó correctamente el registro.'
        ]);
    }
}
