<?php

namespace ILC\CargaArchivos\Traits;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

trait HandlesRelatedTables {

    /**
     * Obtiene las tablas relacionadas y sus campos.
     *
     * @param string $table
     * @param bool $withExplicitRelationships
     * @return array
     * @throws \Exception
     */
    public function getRelatedTablesWithFields(string $table, bool $withExplicitRelationships = false): array
    {
        if (!Schema::hasTable($table)) {
            throw new \Exception("La tabla '{$table}' no existe en la base de datos.");
        }

        try {
            $query = "
            SELECT
                tc.constraint_name,
                kcu.column_name AS foreign_column,
                ccu.table_name AS related_table,
                ccu.column_name AS related_column
            FROM
                information_schema.table_constraints AS tc
                JOIN information_schema.key_column_usage AS kcu
                ON tc.constraint_name = kcu.constraint_name
                AND tc.table_schema = kcu.table_schema
                JOIN information_schema.constraint_column_usage AS ccu
                ON ccu.constraint_name = tc.constraint_name
                AND ccu.table_schema = tc.table_schema
            WHERE
                tc.constraint_type = 'FOREIGN KEY' AND tc.table_name = ?
        ";

            $foreignKeys = DB::select($query, [$table]);
            $relatedTables = [];

            $categorizedTypes = [
                'alfabetico' => ['character varying', 'text', 'varchar', 'char', 'string'],
                'numerico' => ['bigint', 'int', 'integer', 'smallint', 'decimal', 'numeric', 'float', 'double'],
                'verdadero/falso' => ['boolean', 'bool'],
                'fecha' => ['timestamp', 'date', 'datetime', 'timestamp without time zone', 'time']
            ];


            foreach ($foreignKeys as $foreignKey) {
                $relatedTableName = $foreignKey->related_table;

                if (!Schema::hasTable($relatedTableName)) {
                    continue;
                }

                if ($withExplicitRelationships) {
                    $relatedFields = $this->getTableFieldsWithDataType($relatedTableName);
                    $excludedFields = $this->getExcludedFields($relatedTableName);
                    $fields = array_filter($relatedFields, function ($column) use ($excludedFields) {
                        return !in_array($column['nombre'], $excludedFields);
                    });
                } else {
                    $relatedFields = Schema::getColumnListing($relatedTableName);
                    $tableFields = array_values(array_diff($relatedFields, $this->getExcludedFields($relatedTableName)));
                    $fields = array_map(fn($field) => $field, $tableFields);
                }

                if ($withExplicitRelationships) {
                    $fields = array_map(function ($column) use ($categorizedTypes) {
                        foreach ($categorizedTypes as $category => $types) {
                            if (in_array($column['tipo'], $types)) {
                                $column['tipo'] = $category;
                                return $column;
                            }
                        }
                        $column['tipo'] = 'alfabetico';
                        return $column;
                    }, $fields);
                }



                $fields = array_values($fields);


                $relatedTables[] = [
                    'tabla' => $relatedTableName,
                    'campos' => $fields,
                ];
            }

            return $relatedTables;
        } catch (\Exception $e) {
            throw new \Exception("Error al obtener las relaciones de la tabla '{$table}': " . $e->getMessage());
        }
    }

    private function getDatabaseFields(string $tableName): array
    {
        if (!Schema::hasTable($tableName)) {
            throw new \Exception("La tabla '{$tableName}' no existe en la base de datos.");
        }

        $fields = Schema::getColumnListing($tableName);

        $excludedFields = ['created_at', 'updated_at'];
        return array_values(array_diff($fields, $excludedFields));
    }

    private function getTableFieldsWithDataType(string $table): array
    {
        if (!Schema::hasTable($table)) {
            throw new \Exception("La tabla '{$table}' no existe en la base de datos.");
        }

        $columns = Schema::getColumnListing($table);
        $fields = [];

        $categorizedTypes = [
            'alfabetico' => ['character varying', 'text', 'varchar', 'char', 'string'],
            'numerico' => ['bigint', 'int', 'integer', 'smallint', 'decimal', 'numeric', 'float', 'double'],
            'verdadero/falso' => ['boolean', 'bool'],
            'fecha' => ['timestamp', 'date', 'datetime', 'timestamp without time zone', 'time']
        ];

        foreach ($columns as $column) {
            $fieldDetails = DB::select("
                SELECT column_name, data_type, is_nullable
                FROM information_schema.columns
                WHERE table_name = ? AND column_name = ?
            ", [$table, $column])[0];

            $dataType = 'alfabetico';
            foreach ($categorizedTypes as $cat => $types) {
                if (in_array($fieldDetails->data_type, $types)) {
                    $dataType = $cat;
                    break;
                }
            }

            $fields[] = [
                'nombre' => $fieldDetails->column_name,
                'tipo' => $dataType,
                'requerido' => $fieldDetails->is_nullable === 'NO',
                'tabla' => $table,
            ];
        }

        return $fields;
    }


    /**
     * Obtiene los campos excluidos de una tabla
     *
     * @param string $tableName
     * @return array
     */
    function getExcludedFields(string $tableName): array
    {
        $excludedFields = [
            'users' => ['id','created_at', 'updated_at', 'email_verified_at', 'remember_token'],
            'perfiles' => ['id', 'user_id', 'created_at', 'updated_at'],
            'roles' => ['id','created_at', 'updated_at'],
            'default' => ['id','created_at', 'updated_at'],
        ];

        return $excludedFields[$tableName] ?? $excludedFields['default'];
    }

    public function getAllTablesInSchema(): array
    {
        $tables = DB::select("
                SELECT table_name
                FROM information_schema.tables
                WHERE table_schema = 'public'
                ORDER BY table_name
            ");

        return array_map(fn($table) => $table->table_name, $tables);
    }
}
