<?php

namespace ILC\CargaArchivos\Services;

 use ILC\CargaArchivos\Models\ILCCargaArchivo;
 use Illuminate\Support\Facades\Log;
 use Illuminate\Support\Str;
 use Maatwebsite\Excel\Facades\Excel;
 use Illuminate\Http\UploadedFile;
 use PhpOffice\PhpSpreadsheet\IOFactory;

 class CargaArchivos {
    protected $definition;
    protected UploadedFile $file;

     /**
      * @param UploadedFile $file
      */
     public function __construct($definition, UploadedFile $file)
     {
         $this->definition = $definition;
         $this->file = $file;
     }


     /**
      * @param string|null $slug
      * @param UploadedFile $file
      * @return static
      */
     public static function archivo(?string $slug, UploadedFile $file): static
     {
         $fileDefinition = $slug ? ILCCargaArchivo::where('referencia_nombre', $slug)->first() : null;
         $self = new static($fileDefinition, $file);

         if (!$fileDefinition) {
             $self->definition = $self->createFileStructureFromFile($file);
         }

         return $self;
     }


     /**
      * @return array
      */
     public function get(): array
     {
         return $this->buildFileStructure();
     }


     /**
      * @return string|false
      */
     public function getJson(): string|false
     {
         return json_encode($this->buildFileStructure());
     }


     /**
      * @return array
      */
     protected function buildFileStructure(): array
     {
         $fileMap = [
             'nombre' => '',
             'referencia_nombre' => '',
             'descripcion' => '',
             'mapa_estructura' => [],
             'mapa_modelo' => [],
             'tipo_archivo' => $this->file->extension() == 'xlsx' ? 'excel' : 'csv',
             'created_at' => null,
             'updated_at' => null,
         ];

         if (is_array($this->definition)) {
             $fileMap['mapa_estructura'] = $this->definition;
         } elseif ($this->definition && !is_array($this->definition)) {
             $fileMap['nombre'] = $this->definition->nombre ?? '';
             $fileMap['referencia_nombre'] = $this->definition->referencia_nombre ?? '';
             $fileMap['descripcion'] = $this->definition->descripcion ?? '';
             $fileMap['mapa_estructura'] = json_decode($this->definition->mapa_estructura ?? '{}', true);  // Return as array
             $fileMap['mapa_modelo'] = json_decode($this->definition->mapa_modelo ?? '{}', true);  // Return as array
             $fileMap['created_at'] = $this->definition->created_at ?? null;
             $fileMap['updated_at'] = $this->definition->updated_at ?? null;
         }

         return $fileMap;
     }


     /**
      * @return bool
      */
     public function store(): bool
     {
         $data = $this->buildFileStructure();

         try {
             $this->storeFileDefinition($data);

             return true;
         } catch (\Exception $e) {
             Log::error('Error al tratar de guardar la definición del archivo: '. $e->getMessage());

             return false;
         }
     }

     protected function storeFileDefinition($sheet): void
    {

        if (is_array($sheet['mapa_estructura'])) {
            $sheet['mapa_estructura'] = json_encode($sheet['mapa_estructura']);
        }

        if (is_array($sheet['mapa_modelo'])) {
            $sheet['mapa_modelo'] = json_encode($sheet['mapa_modelo']);
        }

        try {
            ILCCargaArchivo::updateOrCreate(
                ['referencia_nombre' => Str::slug($sheet['nombre'])],
                [
                    'nombre' => $sheet['nombre'],
                    'descripcion' => $sheet['descripcion'],
                    'mapa_estructura' => $sheet['mapa_estructura'],
                    'mapa_modelo' => $sheet['mapa_modelo'],
                    'tipo_archivo' => $sheet['tipo_archivo'],
                    'updated_at' => now(),
                ]
            );
        } catch (\Exception $e) {
            Log::error('Error al tratar de guardar el archivo: ' . $e->getMessage());
        }
    }

     protected function createFileStructureFromFile(UploadedFile $file): array
     {
         $spreadsheet = IOFactory::load($file->getPathname());
         $sheetNames = $spreadsheet->getSheetNames();
         $sheets = Excel::toCollection(null, $this->file);

         $fileStructure = [
             'hojas' => []
         ];

         foreach ($sheets as $sheetIndex => $sheet) {
             $columns = [];

             foreach ($sheet->first() as $columnIndex => $columnName) {
                 $columns[] = [
                     'posicion' => $columnIndex,
                     'nombre' => $columnName,
                     'tipo' => null,
                     'requerido' => false
                 ];
             }

             $fileStructure['hojas'][] = [
                 'nombre' => $sheetNames[$sheetIndex],
                 'posicion_hoja' => $sheetIndex,
                 'columnas' => $columns
             ];
         }

         return $fileStructure;
     }
}



