<?php

namespace ILC\CargaArchivos\Exports;

use Carbon\Carbon;
use Generator;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Rap2hpoutre\FastExcel\FastExcel;
use Rap2hpoutre\FastExcel\SheetCollection;
use Symfony\Component\HttpFoundation\StreamedResponse;
class DataExportService
{
    /**
     * Genera el archivo Excel para multiples tablas
     *
     * @param Request $request
     * @param array $modelMap
     * @param string|null $fileName
     * @return StreamedResponse
     */
    public function generateMultiTableExcel(
        Request $request,
        array $modelMap,
        ?string $fileName = null
    ): StreamedResponse
    {
        $sheets = [];

        foreach ($modelMap['tablas'] as $tablaDefinition) {
            $table = $tablaDefinition['tabla'];
            $fields = $tablaDefinition['campos'];

            if (empty($fields)) {
                return response()->json(['error' => "No hay campos definidos para la tabla $table"], 400);
            }

            $tableGenerator = $this->generateTableData($table, $fields, $tablaDefinition['isEloquent'] ?? false, $request->get('filtros', []));
            $sheets[$table] = $tableGenerator;
        }

        $sheetCollection = new SheetCollection($sheets);
        $timestamp = Carbon::now()->format('YmdHis');

        $fileName = $fileName
            ? preg_replace('/[^a-zA-Z0-9\s._-]/', '', trim($fileName))
            : "Descarga_{$timestamp}";

        $fileName = trim($fileName) . '.xlsx';

        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
            'Content-Transfer-Encoding' => 'binary',
            'X-Content-Type-Options' => 'nosniff',
            'Cache-Control' => 'no-cache, no-store, must-revalidate',
            'Pragma' => 'no-cache',
            'Expires' => '0',
        ];

        return response()->streamDownload(function () use ($sheetCollection) {
            (new FastExcel($sheetCollection))->export('php://output');
        }, $fileName, $headers);
    }


    /**
     * Procesa los campos de una tabla
     *
     * @param string $table
     * @param array $fields
     * @param bool $is_eloquent
     * @return Generator
     */
    protected function generateTableData(string $table, array $fields, bool $is_eloquent = false, array $filtros = []): Generator
    {
        $fieldNames = array_column($fields, 'nombre');
        $query = (! $is_eloquent) ? DB::table($table)->select($fieldNames) : ('App\\Models\\'.$table)::query();

        // Aplicación de filtros.
        foreach ($filtros as $filtro_key => $filtro_value) {
            if (is_array($filtro_value)) {
                $query = $query->whereIn($filtro_key, $filtro_value);
            } else {
                $query = $query->where($filtro_key, $filtro_value);
            }
        }

        foreach ($query->cursor() as $row) {
            $rowArray = (! $is_eloquent) ? (array) $row : $row;
            $formattedRow = [];

            foreach ($fields as $field) {
                $fieldName = $field['nombre'];
                $fieldNameDisplay = $field['apodo'];
                $fieldType = $field['tipo'] ?? 'alfabetico';
                $value = $rowArray[$fieldName] ?? data_get($rowArray, $fieldName) ?? null;

                switch ($fieldType) {
                    case 'fecha':
                        $formattedRow[$fieldNameDisplay] = $value ? Carbon::parse($value)->format('Y-m-d') : null;
                        break;
                    case 'verdadero/falso':
                        $formattedRow[$fieldNameDisplay] = is_null($value) ? null : ($value ? 'Sí' : 'No');
                        break;
                    case 'numerico':
                        $formattedRow[$fieldNameDisplay] = is_null($value) ? null : (int)$value;
                        break;
                    case 'alfabetico':
                    default:
                        $formattedRow[$fieldNameDisplay] = (string)$value;
                }
            }

            yield $formattedRow;
        }
    }
}
