<?php

namespace ILC\CargaArchivos\Services;

use ILC\CargaArchivos\Imports\DataImport;
use ILC\CargaArchivos\Models\ILCCargaArchivo;
use ILC\CargaArchivos\Traits\HandlesFileStructure;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Rap2hpoutre\FastExcel\FastExcel;

class CargaArchivos {

    use HandlesFileStructure, HandlesRelatedTables;
    protected $definition;
    protected DataImport $dataImport;

    /**
     * @param string|null $definition
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @throws \Exception
     */
    public function __construct(?string $definition, UploadedFile $file, int $headersLineNumber)
    {
        $this->file = $file;
        $this->headersLineNumber = $headersLineNumber;
        $this->setFileDefinition($definition);
        $this->dataImport = new DataImport($file, $headersLineNumber, $this->structureMap, $this->modelMap, $this->tableName);
    }

    /**
     * Retorna el no. de línea donde se encuentran los encabezados
     *
     * @return int
     */
    public function headingRow(): int
    {
        return $this->headersLineNumber;
    }

    /**
     * Retorna la definición de la carga del archivo.
     *
     * @param string|null $slug
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @return static
     * @throws \Exception
     */
    public static function archivo(?string $slug, UploadedFile $file, int $headersLineNumber): static
    {
        if (!$slug) {
            throw new \Exception('La referencia no puede estar vacía.');
        }

        $fileDefinition = ILCCargaArchivo::where('referencia_nombre', $slug)->first();

        if ($fileDefinition) {
            return new static($fileDefinition->referencia_nombre, $file, $headersLineNumber);
        }
        throw new \Exception("No se encontró la definición del archivo con la referencia '{$slug}'");
    }

    /**
     * Retorna la estructura del Archivo que se desea importar usando FastExcel.
     *
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @return array
     * @throws \Exception
     */
    public static function getFileStructure(UploadedFile $file, int $headersLineNumber = 1): array
    {
        try {
            return (new static(null, $file, $headersLineNumber))->createFileStructureFromFile($file, $headersLineNumber);
        } catch (\Exception $e) {
            throw new \Exception("Error al obtener la estructura del archivo: " . $e->getMessage());
        }
    }

    /**
     * Retorna la estructura del archivo como arreglo.
     *
     * @return array
     */
    public function toArray(): array
    {
        return $this->buildFileStructure();
    }

    /**
     * Retorna la estructura del archivo como JSON
     *
     * @return string|false
     */
    public function toJson(): string|false
    {
        return json_encode($this->buildFileStructure());
    }

    /**
     * Usa el servicio para retornar los datos del archivo que se va a importar como arreglo.
     *
     * @return array
     */
    public function get(): array
    {
        return $this->dataImport->get();
    }

    /**
     * Usa el servicio para retornar los datos del archivo que se va a importar como JSON.
     *
     * @return string
     */
    public function getJson(): string
    {
        return $this->dataImport->getJson();
    }

    /**
     * Usa el servicio para guardar los registros importados.
     *
     * @return array
     */
    public function store(): array
    {
        return $this->dataImport->store();
    }
}
