<?php

namespace ILC\CargaArchivos\Rules;

use Illuminate\Contracts\Validation\Rule;

class Curp implements Rule
{
    private $error_msg = 'CURP no válida.';

    /**
     * Create a new rule instance.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param  string  $attribute
     * @param  mixed  $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        // Se convierten todas las letras a mayúsculas (para no existir conflicto con minúsculas).
        $value = strtoupper($value);

        // Longitud de la CURP.
        if (strlen($value) != 18) {
            $this->error_msg = 'La CURP debe constar de 18 posiciones.';
            return false;
        }

        // Formato de la CURP.
        $reg = "/([A-Z]{4})?(\d{6})?([HMX])?([A-Z]{2})?([B-DF-HJ-NP-TV-Z]{3})?([A-Z0-9])?([0-9])?/";
        $m = preg_match($reg, $value, $matches);

        if (count($matches) < 2 || $matches[1] == NULL) {
            $this->error_msg = 'Las primeras 4 posiciones de la CURP deben ser letras.';
            return false;
        }

        elseif (count($matches) < 3 || $matches[2] == NULL) {
            $this->error_msg = 'Las posiciones 5 a 10 de la CURP deben ser dígitos.';
            return false;
        }

        elseif (count($matches) < 4 || $matches[3] == NULL) {
            $this->error_msg = 'La posición 11 de la CURP debe ser la letra H, M ó X).';
            return false;
        }

        elseif (count($matches) < 5 || $matches[4] == NULL) {
            $this->error_msg = 'Las posiciones 12 a 13 de la CURP deben ser letras.';
            return false;
        }

        elseif (count($matches) < 6 || $matches[5] == NULL) {
            $this->error_msg = 'Las posiciones 14 a 16 de la CURP deben ser letras consonantes.';
            return false;
        }

        elseif (count($matches) < 7 || $matches[6] == NULL) {
            $this->error_msg = 'La penúltima posición de la CURP debe ser una letra o un dígito.';
            return false;
        }

        elseif (count($matches) < 8 || $matches[7] == NULL) {
            $this->error_msg = 'La última posición de la CURP debe ser un dígito.';
            return false;
        }

        elseif (!$m)
            return false;

        // Dígito verificador de la CURP.
        $digito = $this->verifica($value);
        if(!($digito == substr($value, 17, 1))) {
            $this->error_msg = "El dígito verificador de la CURP no es válido.";
            return false;
        }

        // Si lo anterior no falla, la CURP es válida.
        return true;

    }

    /**
     * Determina el dígito verificador de la CURP dada.
     *
     * @param  string $curp CURP a 18 posiciones.
     * @return int Dígito verificador de la CURP.
     */
    private function verifica($curp)
    {
        $segRaiz      = substr($curp,0,17); //.substring(0,17);
        $chrCaracter  = utf8_decode("0123456789ABCDEFGHIJKLMNÑOPQRSTUVWXYZ");
        $lngSuma = 0.0;
        $lngDigito = 0.0;

        for($i=0; $i<17; $i++) {
            for($j=0;$j<37; $j++) {
                if(substr($segRaiz,$i,1)==substr($chrCaracter,$j,1)) {
                    $intFactor[$i]=$j;
                }
            }
        }

        for($k = 0; $k < 17; $k++) {
            $lngSuma= $lngSuma + ((int)($intFactor[$k]) * (18 - $k));
        }

        $lngDigito= (10 - ($lngSuma % 10));

        if($lngDigito==10) {
            $lngDigito=0;
        }

        return $lngDigito;
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return $this->error_msg;
    }
}
