<?php

namespace ILC\CargaArchivos\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class TemplateDefinitionRequest extends FormRequest
{
    /**
     * Determina si el usuario tiene permitido realizar la acción.
     */
    public function authorize(): bool
    {
        return true;
    }

    protected function prepareForValidation(): void
    {
        $mapaModelo = $this->input('mapa_modelo');

        if (is_string($mapaModelo)) {
            $mapaModelo = json_decode($mapaModelo, true);
        }


        $this->merge([
            'mapa_modelo' => $mapaModelo,
        ]);
    }

    /**
     * Las reglas de validación para la petición.
     *
     * @return array
     */
    public function rules(): array
    {
        return [
            'nombre' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'mapa_modelo' => 'required|array|min:1',
            'mapa_modelo.tablas' => 'required|array|min:1',
            'mapa_modelo.tablas.*.tabla' => 'required|string|max:255',
            'mapa_modelo.tablas.*.campos' => 'required|array|min:1',
            'mapa_modelo.tablas.*.campos.*.nombre' => 'required|string|max:255',
            'mapa_modelo.tablas.*.campos.*.apodo' => 'string|max:255',
            'mapa_modelo.tablas.*.campos.*.tipo' => [
                'required',
                'string',
                Rule::in(['numerico', 'alfabetico', 'fecha', 'verdadero/falso']),
            ],
            'mapa_modelo.tablas.*.campos.*.requerido' => 'required|boolean',
            'mapa_modelo.tablas.*.campos.*.forTemplate' => 'required|boolean',
        ];
    }

    /**
     * Mensajes personalizados de error
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'mapa_modelo.tablas.*.tabla.required' => 'El nombre de la tabla es obligatorio.',
            'mapa_modelo.tablas.*.campos.*.nombre.required' => 'Cada campo debe tener un nombre.',
            'mapa_modelo.tablas.*.campos.*.tipo.required' => 'Cada campo debe tener un tipo.',
            'mapa_modelo.tablas.*.campos.*.tipo.in' => 'El tipo de campo no es válido.',
            'mapa_modelo.tablas.*.campos.*.requerido.required' => 'Debe especificar si el campo es requerido.',
        ];
    }

    /**
     * Se ejecuta una vez que la validación fue exitosa.
     */
    protected function passedValidation(): void
    {
        $processedMap = collect($this->input('mapa_modelo.tablas'))->map(function ($tabla) {
            return [
                'tabla' => $tabla['tabla'],
                'campos' => collect($tabla['campos'])->map(function ($campo) {
                    return [
                        'nombre' => $campo['nombre'],
                        'apodo' => $campo['apodo'] ?? $campo['nombre'],
                        'tipo' => $campo['tipo'],
                        'requerido' => $campo['requerido'] ?? false,
                        'forTemplate' => $campo['forTemplate'] ?? false,
                    ];
                })->toArray(),
            ];
        })->toArray();

        $this->replace([
            'nombre' => $this->input('nombre'),
            'referencia_nombre' => Str::slug($this->input('nombre')),
            'descripcion' => $this->input('descripcion', ''),
            'mapa_modelo' => [
                'tablas' => $processedMap,
            ],
        ]);
    }
}
