<?php

namespace ILC\CargaArchivos\Exports;

use Carbon\Carbon;
use Generator;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Rap2hpoutre\FastExcel\FastExcel;
use Rap2hpoutre\FastExcel\SheetCollection;
use Symfony\Component\HttpFoundation\StreamedResponse;
class DataExportService
{
    /**
     * Genera el archivo Excel para multiples tablas
     *
     * @param Request $request
     * @param array $modelMap
     * @param string|null $fileName
     * @return StreamedResponse
     */
    public function generateMultiTableExcel(
        Request $request,
        array $modelMap,
        ?string $fileName = null
    ): StreamedResponse
    {
        $sheets = [];

        foreach ($modelMap['tablas'] as $tablaDefinition) {
            $table = $tablaDefinition['tabla'];
            $fields = $tablaDefinition['campos'];

            if (empty($fields)) {
                return response()->json(['error' => "No hay campos definidos para la tabla $table"], 400);
            }

            $tableGenerator = $this->generateTableData($table, $fields);
            $sheets[$table] = $tableGenerator;
        }

        $sheetCollection = new SheetCollection($sheets);
        $timestamp = Carbon::now()->format('YmdHis');

        $fileName = $fileName
            ? preg_replace('/[^a-zA-Z0-9\s._-]/', '', trim($fileName))
            : "Descarga_{$timestamp}";

        $fileName = trim($fileName) . '.xlsx';

        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
            'Content-Transfer-Encoding' => 'binary',
            'X-Content-Type-Options' => 'nosniff',
            'Cache-Control' => 'no-cache, no-store, must-revalidate',
            'Pragma' => 'no-cache',
            'Expires' => '0',
        ];

        return response()->streamDownload(function () use ($sheetCollection) {
            (new FastExcel($sheetCollection))->export('php://output');
        }, $fileName, $headers);
    }


    /**
     * Procesa los campos de una tabla
     *
     * @param string $table
     * @param array $fields
     * @return Generator
     */
    protected function generateTableData(string $table, array $fields): Generator
    {
        $fieldNames = array_column($fields, 'nombre');
        $query = DB::table($table)->select($fieldNames);

        foreach ($query->cursor() as $row) {
            $rowArray = (array) $row;
            $formattedRow = [];

            foreach ($fields as $field) {
                $fieldName = $field['apodo'];
                $fieldType = $field['tipo'] ?? 'alfabetico';
                $value = $rowArray[$field['nombre']] ?? null;

                switch ($fieldType) {
                    case 'fecha':
                        $formattedRow[$fieldName] = $value ? Carbon::parse($value)->format('Y-m-d') : null;
                        break;
                    case 'verdadero/falso':
                        $formattedRow[$fieldName] = $value ? 'Sí' : 'No';
                        break;
                    case 'numerico':
                        $formattedRow[$fieldName] = is_null($value) ? null : (int)$value;
                        break;
                    case 'alfabetico':
                    default:
                        $formattedRow[$fieldName] = (string)$value;
                }
            }

            yield $formattedRow;
        }
    }
}
