<?php

namespace ILC\CargaArchivos\Traits;

use ILC\CargaArchivos\Filters\ReadExcelHeadingsFilter;
use ILC\CargaArchivos\Models\ILCCargaArchivo;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Rap2hpoutre\FastExcel\FastExcel;
use Exception;

trait HandlesFileStructure {
    protected UploadedFile $file;
    protected int $headersLineNumber;
    protected array $structureMap = [];
    protected array $modelMap = [];
    protected string $tableName = '';

    /**
     * Crea la estructura basada en la definición del archivo.
     *
     * @param $definition
     * @throws Exception
     */
    protected function setFileDefinition($definition): void
    {
        if ($definition) {
            $fileDefinition = ILCCargaArchivo::where('referencia_nombre', $definition)->first();

            if ($fileDefinition) {
                $this->structureMap = json_decode($fileDefinition->mapa_estructura, true) ?? [];
                $this->modelMap = json_decode($fileDefinition->mapa_modelo, true) ?? [];
                $this->tableName = $this->modelMap['hojas'][0]['tabla'] ?? '';
            } else {
                throw new Exception('No se encontró la definición del archivo con esa referencia.');
            }
        }
    }

    /**
     * Genera la estructura del archivo para guardar su definición.
     *
     * @return array
     */
    protected function buildFileStructure(): array
    {
        $fileMap = [
            'nombre' => '',
            'referencia_nombre' => '',
            'descripcion' => '',
            'mapa_estructura' => [],
            'mapa_modelo' => [],
            'tipo_archivo' => $this->file->extension() == 'xlsx' ? 'excel' : 'csv',
            'created_at' => null,
            'updated_at' => null,
        ];

        if (is_array($this->definition)) {
            $fileMap['mapa_estructura'] = $this->definition;
        } elseif ($this->definition && !is_array($this->definition)) {
            $fileMap['nombre'] = $this->definition->nombre ?? '';
            $fileMap['referencia_nombre'] = $this->definition->referencia_nombre ?? '';
            $fileMap['descripcion'] = $this->definition->descripcion ?? '';
            $fileMap['mapa_estructura'] = json_decode($this->definition->mapa_estructura ?? '{}', true);
            $fileMap['mapa_modelo'] = json_decode($this->definition->mapa_modelo ?? '{}', true);
            $fileMap['created_at'] = $this->definition->created_at ?? null;
            $fileMap['updated_at'] = $this->definition->updated_at ?? null;
        }

        return $fileMap;
    }

    /**
     * Crea la estructura del archivo desde su contenido usando FastExcel.
     *
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @return array
     * @throws Exception
     */
    public function createFileStructureFromFile(UploadedFile $file, int $headersLineNumber = 1): array
    {
        try {
            $fileType = $file->getClientOriginalExtension();
            $isCSV = strtolower($fileType) === 'csv';
            $path = 'imports/temp_' . uniqid() . '.' . $fileType;
            Storage::put($path, file_get_contents($file->getRealPath()));
            $filePath = Storage::path($path);

            $fastExcel = new FastExcel;

            if ($isCSV) {
                $fileName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                $fastExcel->configureCsv(',', '"', '\\', 0);

                $sampleData = [];
                $rowCount = 0;
                $headers = [];

                foreach ($fastExcel->import($filePath) as $rowIndex => $row) {
                    if ($rowIndex + 1 < $headersLineNumber) {
                        continue;
                    }

                    if ($rowIndex + 1 == $headersLineNumber) {
                        $headers = array_keys($row);
                        continue;
                    }

                    $sampleData[] = $row;
                    $rowCount++;

                    if ($rowCount >= 10) {
                        break;
                    }
                }

                $columns = $this->processHeadersFromFastExcel($headers, $sampleData, $headersLineNumber);

                $tableFields = Schema::getColumnListing($fileName);
                $tableFields = array_values(
                    array_diff($tableFields, $this->getExcludedColumns($fileName))
                );

                $relatedTableFields = $this->getRelatedTablesWithFields($fileName);
                $relatedFieldsWithTables = [];

                foreach ($relatedTableFields as $relation) {
                    foreach ($relation['campos'] as $relatedField) {
                        $relatedFieldsWithTables[] = [
                            'campo' => $relatedField,
                            'tabla' => $relation['tabla'],
                        ];
                    }
                }

                $mergedFields = array_map(fn($field) => ['campo' => $field, 'tabla' => $fileName], $tableFields);
                foreach ($relatedFieldsWithTables as $relatedField) {
                    $mergedFields[] = $relatedField;
                }

                $fileStructure = [
                    'hojas' => [
                        [
                            'nombre' => $fileName,
                            'posicion_hoja' => 0,
                            'columnas' => $columns,
                            'campos_tabla' => $mergedFields,
                            'campos_relaciones' => $relatedTableFields,
                        ]
                    ]
                ];

            } else {
                $reader = IOFactory::createReader('Xlsx');
                $reader->setReadDataOnly(true);

                // usamos filtro para evitar cargar el archivo completo en memoria
                // solo se cargan los encabezados y 10 registros para tratar de inferir los tipos de datos en las columnas
                $reader->setReadFilter(new ReadExcelHeadingsFilter($headersLineNumber + 10));
                $spreadsheet = $reader->load($filePath);
                $sheetNames = $spreadsheet->getSheetNames();
                $spreadsheet->disconnectWorksheets();
                unset($spreadsheet);

                $fileStructure = [
                    'hojas' => [],
                ];

                foreach ($sheetNames as $sheetIndex => $sheetName) {
                    $sampleData = [];
                    $rowCount = 0;
                    $headers = [];

                    foreach ($fastExcel->sheet($sheetIndex + 1)->import($filePath, null, 0) as $rowIndex => $row) {
                        if ($rowIndex + 1 < $headersLineNumber) {
                            continue;
                        }

                        if ($rowIndex + 1 == $headersLineNumber) {
                            $headers = array_keys($row);
                            continue;
                        }

                        $sampleData[] = $row;
                        $rowCount++;

                        if ($rowCount >= 10) {
                            break;
                        }
                    }

                    $columns = $this->processHeadersFromFastExcel($headers, $sampleData, $headersLineNumber);

                    $tableFields = Schema::getColumnListing($sheetName);
                    $tableFields = array_values(
                        array_diff($tableFields, $this->getExcludedColumns($sheetName))
                    );

                    $relatedTableFields = $this->getRelatedTablesWithFields($sheetName);
                    $relatedFieldsWithTables = [];

                    foreach ($relatedTableFields as $relation) {
                        foreach ($relation['campos'] as $relatedField) {
                            $relatedFieldsWithTables[] = [
                                'campo' => $relatedField,
                                'tabla' => $relation['tabla'],
                            ];
                        }
                    }

                    $mergedFields = array_map(
                        fn($field) => ['campo' => $field, 'tabla' => $sheetName],
                        $tableFields
                    );

                    foreach ($relatedFieldsWithTables as $relatedField) {
                        $mergedFields[] = $relatedField;
                    }

                    $fileStructure['hojas'][] = [
                        'nombre' => $sheetName,
                        'posicion_hoja' => $sheetIndex,
                        'columnas' => $columns,
                        'campos_tabla' => $mergedFields,
                        'campos_relaciones' => $relatedTableFields,
                    ];
                }
            }

            Storage::delete($path);

            return $fileStructure;
        } catch (Exception $e) {
            if (isset($path) && Storage::exists($path)) {
                Storage::delete($path);
            }
            throw new Exception("Error al crear la estructura del archivo: " . $e->getMessage());
        }
    }

    /**
     * Procesa los encabezados del archivo excel y trata de inferir el tipo de dato de la columna
     *
     * @param array $headers
     * @param array $sampleData
     * @param int $headersLineNumber
     * @return array
     */
    private function processHeadersFromFastExcel(array $headers, array $sampleData, int $headersLineNumber): array
    {
        $columns = [];

        foreach ($headers as $columnIndex => $columnName) {
            $column = Coordinate::stringFromColumnIndex($columnIndex + 1);
            $row = $headersLineNumber;
            $headerColumnRow = "{$column}{$row} - {$columnName}";

            $dataFromColumn = [];
            foreach ($sampleData as $sampleRow) {
                if (isset($sampleRow[$columnName]) && $sampleRow[$columnName] !== null && $sampleRow[$columnName] !== '') {
                    $dataFromColumn[] = $sampleRow[$columnName];
                }
            }

            $typeOfData = $this->determineColumnDataType($dataFromColumn);

            $columns[] = [
                'posicion' => $columnIndex,
                'nombre' => $headerColumnRow,
                'tipo' => $typeOfData,
                'requerido' => false,
            ];
        }

        return $columns;
    }

    /**
     * Trata de inferir el tipo de dato de la columna
     *
     * @param array $dataFromColumn
     * @return string
     */
    protected function determineColumnDataType(array $dataFromColumn): string
    {
        if (empty($dataFromColumn)) {
            return 'string';
        }

        $isBoolean = count(array_filter($dataFromColumn, function($value) {
                return !in_array(strtolower(trim($value)), ['1', '0', 'true', 'false', 'yes', 'no', 'si', 'sí', 'no'], true);
            })) === 0;

        $isInteger = count(array_filter($dataFromColumn, function($value) {
                return !is_numeric($value) || intval($value) != $value;
            })) === 0;

        $isDate = count(array_filter($dataFromColumn, function($value) {
                return strtotime($value) === false;
            })) === 0;

        if ($isBoolean) {
            return 'boolean';
        } elseif ($isInteger) {
            return 'integer';
        } elseif ($isDate) {
            return 'date';
        } else {
            return 'string';
        }
    }
}
