<?php

/**
 * @license Apache 2.0
 */

namespace ILC\CargaArchivos\Http\Controllers;

use ILC\CargaArchivos\Exceptions\AuthorizationException;
use ILC\CargaArchivos\Http\Requests\DynamicCatalogRequest;
use ILC\CargaArchivos\Services\SchemaService;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class JsonSchemaFormController extends Controller
{
    use HandlesRelatedTables;

    protected SchemaService $schemaService;

    public function __construct(SchemaService $schemaService)
    {
        $this->schemaService = $schemaService;
    }

    /**
     * @OA\Post(
     *     path="/generate-schema",
     *     operationId="generateSchema",
     *     tags={"Schema"},
     *     summary="Genera JSON Schema para la tabla solicitada",
     *     description="Accepta los campos de una tabla y retorna un JSON schema.",
     *     @OA\RequestBody(
     *         required=true,
     *         description="Un JSON string que contiene los campos que deben ser procesados.",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="fields", type="string", description="Un JSON string que contiene los campos que deben ser procesados.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="El JSON schema correctamente generado.",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="type", type="string", example="object"),
     *             @OA\Property(
     *                 property="properties",
     *                 type="object",
     *                 @OA\Property(
     *                     property="name",
     *                     type="object",
     *                     @OA\Property(property="type", type="string", example="string"),
     *                     @OA\Property(property="title", type="string", example="name"),
     *                     @OA\Property(property="$id", type="string", example="roles.name"),
     *                     @OA\Property(property="description", type="string", example="Campo de tipo string en la tabla 'roles'."),
     *                     @OA\Property(property="maxLength", type="integer", example=255),
     *                     @OA\Property(property="unique", type="boolean", example=true)
     *                 ),
     *                 @OA\Property(
     *                     property="guard_name",
     *                     type="object",
     *                     @OA\Property(property="type", type="string", example="string"),
     *                     @OA\Property(property="title", type="string", example="guard_name"),
     *                     @OA\Property(property="$id", type="string", example="roles.guard_name"),
     *                     @OA\Property(property="description", type="string", example="Campo de tipo string en la tabla 'roles'."),
     *                     @OA\Property(property="maxLength", type="integer", example=255),
     *                     @OA\Property(property="unique", type="boolean", example=true)
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="required",
     *                 type="array",
     *                 @OA\Items(type="string"),
     *                 example={"name", "guard_name"}
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Los datos de entrada son incorrectos.",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="error", type="string", description="El mensaje de error describiendo el problema.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Error de servidor.",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="error", type="string", description="El mensaje de error generado.")
     *         )
     *     )
     * )
     */

    public function generateSchema(Request $request)
    {
        try {
            $fields = $request->input('fields');

            if (is_string($fields)) {
                $fields = json_decode($fields, true);

                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new \Exception("El formato de json es inválido");
                }
            }

            if (!is_array($fields)) {
                throw new \Exception("Los campos deberían ser un arreglo.");
            }

            $requiredFields = [];
            $context = [];

            foreach ($fields as $field) {
                if (!is_array($field)) {
                    continue;
                }

                if (!empty($field['required'])) {
                    $requiredFields[] = $field['title'];
                }

                $propertySchema = [
                    'type' => $field['type'] ?? 'string',
                    'title' => $field['title'] ?? 'Untitled',
                    'description' => $field['description'] ?? null,
                ];

                foreach ($field as $key => $value) {
                    if (!in_array($key, ['title', 'description', 'type', 'required'], true)) {
                        $propertySchema[$key] = $value;
                    }
                }

                $context[$field['title']] = $propertySchema;
            }

            $schemaService = new SchemaService();
            $schemaJson = json_decode($schemaService->create($context), true);

            if (!empty($requiredFields)) {
                $schemaJson['required'] = $requiredFields;
            }

            return $this->sendResponse(['data' => $schemaJson]);
        } catch (\Exception $e) {
            return $this->sendError('Error al generar el schema: ' . $e->getMessage(), 400);
        }
    }


    /**
     * Obtiene los campos para el JSON schema de una tabla en específico.
     *
     * @param string $table
     * @return JsonResponse
     */
    public function getFieldsForSchema(string $table): JsonResponse
    {
        if (!$this->hasPermission('editar-catalogos-dinamicos')) {
            throw new AuthorizationException();
        }
        return $this->sendResponse(['data' => $this->getFieldsForJsonSchema($table)]);
    }

}
