<?php

namespace ILC\CargaArchivos\Imports;

use Exception;
use ILC\CargaArchivos\Traits\DatabaseMappable;
use ILC\CargaArchivos\Traits\HandlesImportedDataValidation;
use ILC\CargaArchivos\Traits\HandlesRelatedTables;
use ILC\CargaArchivos\Traits\ProgressTrackable;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithUpserts;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Role;
use App\Models\User;
use ILC\AdminUsuarios\Models\Perfil;
use Illuminate\Http\JsonResponse;
use ILC\CargaArchivos\Events\ModelSaved;

class DataImport implements
    ToModel,
    WithValidation,
    WithHeadingRow,
    SkipsEmptyRows,
    WithBatchInserts,
    WithUpserts,
    WithChunkReading,
    SkipsOnFailure,
    SkipsOnError
{
    use Importable, SkipsErrors, SkipsFailures, ProgressTrackable, HandlesImportedDataValidation, DatabaseMappable, HandlesRelatedTables;

    protected array $importDetails = [
        'file_name' => '',
        'total_records_in_file' => 0,
        'records_added' => 0,
        'records_updated' => 0,
        'records_with_errors' => 0,
        'errors_detail' => null,
    ];

    protected array $structureMap;
    protected array $modelMap;
    protected string $tableName;
    protected array $validatedData = [];
    protected array $failedData = [];
    protected int $successfulRegistrations = 0;
    protected int $failedRegistrations = 0;
    protected UploadedFile $file;
    protected int $headersLineNumber;
    protected int $currentRowIndex;
    protected string $currentSheetName;
    protected int $currentSheetIndex = 0;

    /**
     * @param UploadedFile $file
     * @param int $headersLineNumber
     * @param array $structureMap
     * @param array $modelMap
     * @param string $tableName
     */
    public function __construct(UploadedFile $file, int $headersLineNumber, array $structureMap, array $modelMap, string $tableName)
    {
        $this->file = $file;
        $this->headersLineNumber = $headersLineNumber;
        $this->structureMap = $structureMap;
        $this->modelMap = $modelMap;
        $this->tableName = $tableName;
        $this->currentRowIndex = 0;
        $this->currentSheetName = '';
        $this->importDetails['file_name'] = $file->getClientOriginalName();
    }


    public function model(array $row)
    {

        try {
            $tableName = $this->currentSheetName;
            $index = $this->currentSheetIndex;

            if (!$tableName) {
                throw new Exception("No se encontró el nombre de la tabla para la hoja con índice {$this->currentSheetIndex}");
            }

            $modelClass = $this->getModelClassForTable($tableName);

            if (!$modelClass) {
                throw new Exception("No se encontró el modelo para la tabla {$tableName}");
            }

            $modelInstance = new $modelClass;
            $uniqueFields = $this->getUniqueFieldForDatabaseTable($tableName, $row, $index);
            $mappingResult = $this->processFieldMapping($row, $tableName);

            if (!empty($mappingResult['errors'])) {
                $this->currentRowIndex++;
                throw new Exception("Errores de validación: " . implode(', ', $mappingResult['errors']));
            }

            $row = $mappingResult['data'];
            $relatedModel = $row['modelo_relacionado'] ?? null;
            unset($row['modelo_relacionado'], $row['tabla_relacionada']);

            if (!empty($row)) {
                $row['updated_at'] = now();
                $createdModel = $modelInstance->updateOrCreate($uniqueFields, $row);

                if ($createdModel->wasRecentlyCreated) {
                    $this->incrementAddedRows();
                } else {
                    $this->incrementUpdatedRows();
                }

                event(new ModelSaved($createdModel));

                if ($relatedModel) {
                    $relationMethod = Str::camel(class_basename(get_class($relatedModel)));

                    if (method_exists($createdModel, $relationMethod)) {
                        $createdModel->$relationMethod()->associate($relatedModel);
                        $createdModel->save();
                    } else {
                        throw new Exception("No se encontró la relación para el modelo relacionado '{$relationMethod}'.");
                    }
                }

                $this->updateUploadProgress($this->successfulRegistrations, $this->failedRegistrations, $this->importDetails['total_records_in_file']);
                $this->getImportProgress();
            }

        } catch (Exception $e) {
            $this->importDetails['total_records_in_file']++;
            $this->importDetails['records_with_errors']++;
            $this->failedRegistrations++;
            $errors = [$e->getMessage()];

            $this->failedData[] = [
                'hoja' => $this->currentSheetName,
                'fila' => $this->currentRowIndex,
                'errors' => $errors
            ];
            return null;
        }

        return null;
    }

    protected function getModelClassForTable($tableName): ?string
    {
        $models = [
            'users' => User::class,
            'roles' => Role::class,
            'perfiles' => Perfil::class
        ];

        return $models[$tableName] ?? null;
    }

    /**
     * @throws Exception
     */
    private function processFieldMapping(array $row, string $tableName): array
    {
        $errors = [];
        $preparedData = [];
        $columns = Schema::getColumnListing($tableName);
        $fieldsFromRelatedTables = $this->getRelatedTablesWithFields($tableName);

        foreach ($row as $fileField => $value) {
            if (in_array($fileField, $columns)) {
                $preparedData[$fileField] = $value;
            } else {
                $relatedTable = collect($fieldsFromRelatedTables)
                    ->first(fn($table) => in_array($fileField, $table['campos']));

                if ($relatedTable) {
                    $relatedTableName = $relatedTable['tabla'];

                    $modelClass = $this->getModelClassForTable($relatedTableName);

                    if ($modelClass) {
                        $relatedRecord = $modelClass::where($fileField, $value)->first();
                        if ($relatedRecord) {
                            $preparedData['modelo_relacionado'] = $relatedRecord;
                            $preparedData['tabla_relacionada'] = $tableName;
                        } else {
                            $errors[] = "El valor '$value' para el campo '$fileField' no se encuentra en la tabla '$relatedTableName'.";
                        }
                    } else {
                        $errors[] = "No se encontró un modelo para la tabla '$relatedTableName'.";
                    }
                }
            }
        }

        return [
            'errors' => $errors,
            'data' => $preparedData,
        ];
    }

    function getExcludedColumns(string $tableName): array
    {
        $excludedFields = [
            'users' => ['id','created_at', 'updated_at', 'email_verified_at', 'remember_token'],
            'perfiles' => ['id', 'user_id', 'created_at', 'updated_at'],
            'roles' => ['id','created_at', 'updated_at'],
            'default' => ['id','created_at', 'updated_at'],
        ];

        return $excludedFields[$tableName] ?? $excludedFields['default'];
    }

    protected function incrementAddedRows(): void
    {
        $this->importDetails['records_added']++;
        $this->importDetails['total_records_in_file']++;
        Log::info(['registros_agregados', $this->importDetails['records_added']]);
    }

    protected function incrementUpdatedRows(): void
    {
        $this->importDetails['total_records_in_file']++;
        $this->importDetails['records_updated']++;
        Log::info(['registros_actualizados', $this->importDetails['records_updated']]);

    }

    public function getImportDetails(): array
    {
        return $this->importDetails;
    }


    public function getImportProgress(): JsonResponse
    {
        $progress = $this->getUploadProgress();
        Log::info('avance: ', [$progress]);
        return response()->json($progress);
    }

    /**
     * @return int
     */
    public function batchSize(): int
    {
        return 1000;
    }


    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 100;
    }


    /**
     * @return string
     */
    public function uniqueBy(): string
    {
        return 'email';
    }


    /**
     * @return array
     */
    public function rules(): array
    {
        $rules = [];

        if (!empty($this->structureMap)) {

            foreach ($this->structureMap['hojas'][0]['columnas'] as $columna) {

                $ruleSet = $columna['requerido'] ? ['required'] : [];
                if ($columna['tipo'] === 'fecha') {
                    $ruleSet[] = 'date';
                } elseif ($columna['tipo'] === 'alfabetico') {
                    $ruleSet[] = 'string';
                } elseif ($columna['tipo'] === 'numerico') {
                    $ruleSet[] = 'integer';
                } elseif ($columna['tipo'] === 'boolean') {
                    $ruleSet[] = 'boolean';
                }

                $rules[$columna['nombre']] = implode('|', $ruleSet);
            }
        }

        return $rules;
    }


    public function get(): array
    {
        try {
            $sheets = Excel::toCollection(null, $this->file);
            $processedData = [];

            foreach ($sheets as $index => $sheet) {

                if ($sheet->isEmpty()) {
                    continue;
                }

                $sheetStructure = $this->structureMap['hojas'][$index] ?? null;
                $sheetMapping = $this->modelMap['hojas'][$index] ?? null;

                if (!$sheetStructure || !$sheetMapping) {
                    Log::error("No se encontraron definiciones para la hoja con índice {$index}.");
                    continue;
                }

                $rowsAfterHeadings = $sheet->slice($this->headersLineNumber);
                $this->importDetails['total_records'] = count($rowsAfterHeadings);

                foreach ($rowsAfterHeadings as $rowIndex => $row) {

                    $this->currentRowIndex = $rowIndex + 1;
                    $this->currentSheetName = $sheetMapping['nombre'];


                    $mappedRow = $this->mapRowToDatabase($row, $this->currentRowIndex, $index, $sheetStructure, $sheetMapping);

                    if (!empty($mappedRow)) {
                        $this->validatedData[] = $mappedRow;
                        $processedData[$sheetMapping['tabla']][] = $mappedRow;
                        $this->successfulRegistrations++;
                    }
                }
            }

            return [
                'success' => true,
                'data' => $processedData,
                'errors' => $this->failedData,
            ];

        } catch (Exception $e) {
            $this->importDetails['total_records_in_file']++;
            Log::error('Error al importar los datos: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Error al importar',
                'error' => $e->getMessage(),
            ];
        }
    }


    /**
     * Retorna los datos a importar como json.
     *
     * @return string
     * @return string
     */
    public function getJson(): string
    {
        return json_encode($this->get());
    }


    /**
     * Guarda los datos importados.
     *
     * @return array
     */
    public function store(): array
    {
        if (empty($this->validatedData)) {
            $result = $this->get();

            if (!$result['success']) {
                Log::error('No se pudo procesar el archivo: ' . $result['message']);

                return [
                    'success' => false,
                    'errors' => $result['errors'],
                    'message' => 'No se pudo procesar el archivo: ' . $result['message'],
                ];
            }

            $this->validatedData = $result['data'];
            $this->failedData = $result['errors'];
        }

        if (empty($this->validatedData)) {
            Log::warning('Sin datos para exportar.');

            return [
                'success' => false,
                'errors' => $result['errors'] ?? [],
                'message' => 'Sin datos para exportar.',
            ];
        }

        try {

            $this->importDetails['errors_detail'] = $this->importDetails['errors_detail'] ?? [];

            foreach ($this->validatedData as $index => $data) {
                if ($index) {
                    $this->currentSheetName = $index;

                    foreach ($data as $singleRow) {
                        $this->model($singleRow);
                    }

                    $this->currentSheetIndex++;
                } else {
                    Log::error("No se encontró el nombre de la tabla para la hoja {$index}.");
                }
            }

            Log::info('Registros guardados correctamente.');
            $importDetails = $this->getImportDetails();

            return [
                'success' => true,
                'importDetails' => [
                    'file_name' => $importDetails['file_name'],
                    'total_records' => $importDetails['total_records_in_file'],
                    'records_added' => $importDetails['records_added'],
                    'records_updated' => $importDetails['records_updated'],
                    'records_with_errors' => count($this->failedData),
                    'errors_detail' => $this->failedData ?? [],
                    ],
                'message' => $importDetails['records_with_errors'] == 0
                    ? 'Los registros se han guardado correctamente.'
                    : 'Hubo ' . $importDetails['records_with_errors'] . ' errores durante la importación.',
            ];

        } catch (Exception $e) {
            Log::error('Error al guardar los registros: ' . $e->getMessage());

            return [
                'success' => false,
                'errors' => $this->importDetails['errors_detail'] ?? [],
                'message' => 'Error al guardar los registros.',
            ];
        }
    }

    private function getUniqueFieldForDatabaseTable(string $tableName, array $rowData, int $index): array
    {
        $uniqueField = null;
        $columnMapping = $this->structureMap['hojas'][$index]['columnas'] ?? [];

        foreach ($columnMapping as $column) {
            if (isset($column['registro_unico']) && $column['registro_unico'] === true) {
                foreach ($this->modelMap['hojas'][$index]['columnas'] as $modelColumn) {
                    if ($column['posicion'] === $modelColumn['columna_archivo'] && isset($modelColumn['columna_bd'])) {
                        $uniqueField[$modelColumn['columna_bd']] = $rowData[$modelColumn['columna_bd']];
                    }
                }
            }
        }

        return $uniqueField;
    }

    public function getRelatedTablesWithFields(string $table): array
    {
        if (!Schema::hasTable($table)) {
            throw new \Exception("La tabla '{$table}' no existe en la base de datos.");
        }

        try {
            $query = "
            SELECT
                tc.constraint_name,
                kcu.column_name AS foreign_column,
                ccu.table_name AS related_table,
                ccu.column_name AS related_column
            FROM
                information_schema.table_constraints AS tc
                JOIN information_schema.key_column_usage AS kcu
                ON tc.constraint_name = kcu.constraint_name
                AND tc.table_schema = kcu.table_schema
                JOIN information_schema.constraint_column_usage AS ccu
                ON ccu.constraint_name = tc.constraint_name
                AND ccu.table_schema = tc.table_schema
            WHERE
                tc.constraint_type = 'FOREIGN KEY' AND tc.table_name = ?
        ";

            $foreignKeys = DB::select($query, [$table]);
            $relatedTables = [];

            foreach ($foreignKeys as $foreignKey) {
                $relatedTableName = $foreignKey->related_table;

                if (!Schema::hasTable($relatedTableName)) {
                    continue;
                }

                $relatedFields = Schema::getColumnListing($relatedTableName);
                $tableFields = array_values(array_diff($relatedFields, $this->getExcludedColumns($relatedTableName)));
                $fields = array_map(fn($field) => $field, $tableFields);

                $relatedTables[] = [
                    'tabla' => $relatedTableName,
                    'campos' => $fields,
                ];
            }

            return $relatedTables;
        } catch (\Exception $e) {
            throw new \Exception("Error al obtener las relaciones de la tabla '{$table}': " . $e->getMessage());
        }
    }

}
