<?php
namespace ILC\AdminUsuarios\Http\Requests;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use ILC\AdminUsuarios\Helpers\CertHelper;
use ILC\AdminUsuarios\Rules\Curp as CurpRule;
use ILC\AdminUsuarios\Exceptions\CertificateReadException;
use App\Models\User;

class UpdateUserRequest extends FormRequest
{
    protected $stopOnFirstFailure = true;

    /**
     * Determina si el usuario está autorizado para hacer la petición.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return auth()->check() && auth()->user()->hasPermissionTo('edicion-de-usuarios');
    }

    /**
     * Procesa campos antes de validarlos
     *
     * @return void
     */
    protected function prepareForValidation(): void
    {
        $user = User::find($this->user->id);
        $this->merge(['id' => $user->id]);
        $this->processCertificateFile($this, $user);

        if($this->filled('email')){
            $this->merge(['email' => Str::lower($this->email)]);
        }else{
            $this->merge(['email' => $user->email]);
        }
        if($this->filled('curp')){
            $this->merge(['curp_rule' => $this->curp]);
        }else{
            $this->merge(['curp' => $user->curp]);
            $this->merge(['curp_rule' => $user->curp]);
        }
    }

    /**
     * Procesa campos despues de la validación.
     */
    protected function passedValidation(): void
    {
        if($this->filled('password')){
            $secret = Str::random(10);
            $this->merge(['password' => Hash::make($secret), 'secret' => $secret]);
        }
        $this->request->remove('valid');
        $this->request->remove('curp_rule');
        $this->files->remove('certificado');
        $this->convertedFiles = null;
    }
    /**
     * Las reglas de validación para crear usuarios
     *
     * @return array
     */
    public function rules(): array
    {
        $default_rules = [
            'nombre' => ['sometimes','max:255'],
            'primer_apellido' => ['sometimes','max:100'],
            'segundo_apellido' => ['sometimes','max:100'],
            'email' => ['sometimes','max:255','email','unique:users,email,' . $this->user->id],
            'curp' => ['sometimes','min:18','max:18','unique:perfiles,curp,' . $this->user->perfil->id],
            'curp_rule' => [['sometimes', new CurpRule]],
            'password' => ['sometimes','max:10'],
            'roles' => ['sometimes','array','distinct','exists:roles,name'],
            'permisos' => ['sometimes','array','distinct','exists:permissions,name'],
            'serie' => ['sometimes','max:60'],
            'autenticacion' => ['sometimes','in:certificado,ldap,api,default'],
            // 'estado' => 'sometimes|boolean'
        ];

        return array_merge_recursive($default_rules, config('adminusuarios.validation_rules.update', []));
    }


    /**
     * Las traducciones de las reglas
     *
     * @return array
     */
    public function messages(): array
    {
        return trans("adminusuarios::validation");
    }


    /**
     * Las traducciones de los atributos a validar
     *
     * @return array
     */
    public function attributes(): array
    {
        return trans("adminusuarios::validation.attributes");
    }

    /**
     * Procesa el archivo del certificado
     *
     * @param Request $request
     * @return void
     */
    protected function processCertificateFile(Request $request, User $user)
    {
        if ($request->hasFile('certificado')) {
            // $mime = $request->certificado->getClientMimeType();
            // if($mime != 'application/pkix-cert'){
            //     throw new CertificateReadException('El campo certificado debe ser un archivo de tipo: cer', 422);
            // }

            $cert_helper = new CertHelper($request->certificado, $user);
            $datos = $cert_helper->get_datos(true);
            if (count($datos)) {
                $current = date('Y-m-d');
                if($datos['valid'] < $current){
                    throw new CertificateReadException('El certificado no es vigente');
                }
                $request->merge(['autenticacion' => 'certificado']);

                foreach($datos as $key => $value){
                    $request->merge([$key => $value]);
                }
            }else {
                throw new CertificateReadException();
            }
        }
    }

}
