<?php

namespace ILC\AdminUsuarios\Http\Controllers;

use ILC\AdminUsuarios\Auth\AuthManagerContext;
use ILC\AdminUsuarios\Exceptions\InvalidCredentialsException;
use ILC\AdminUsuarios\Exceptions\UserNotFoundException;
use ILC\AdminUsuarios\Http\Requests\ValidateCredentialsRequest;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use ILC\AdminUsuarios\Http\Resources\UserResource;

class AuthController extends Controller
{
    protected AuthManagerContext $authManager;

    public function __construct(AuthManagerContext $authManager)
    {
        $this->authManager = $authManager;
    }

     public function login(Request $request) {
         if(!auth()->check()){
             return $this->sendError(null, 401);
         }
     }

    // public function dashboard(Request $request) {
    //     return $this->sendResponse('success', 200);
    // }

    /**
     * Autentifica al usuario de acuerdo a la estrategia
     *
     * @param ValidateCredentialsRequest $request
     * @return JsonResponse
     */
    public function authenticate(ValidateCredentialsRequest $request): JsonResponse
    {
        try {
            $user = $this->authManager->authenticate($request);

            if (!$user->perfil->estado) {
                auth()->user()->tokens()->delete();
                return $this->sendError('Usuario Inactivo', 403);
            }

            return $this->respondWithToken($user);

        } catch (InvalidCredentialsException $e) {
            return $this->sendError($e->getErrorMessage(), 401);
        } catch (\Exception $e) {
            report($e);
            return $this->sendError($e->getMessage(), 500);
        }
    }

    /**
     * Elimina el token del usuario
     *
     * @return JsonResponse
     */
    public function signout(Request $request): JsonResponse
    {
        try {
            $user = $request->user();

            // Eliminamos el token de acceso actual si se está usando Sanctum
            if (!$user->tokens->isEmpty()) {
                // Eliminar todos los tokens asociados al usuario
                $user->tokens()->delete();
            }

            // Invalida la sesión y regenera el token CSRF
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            // Cierra sesión del guard
            auth()->guard('web')->logout();
            session()->flush();

            return $this->sendResponse('Se ha cerrado la sesión con éxito', 200);
        } catch (UserNotFoundException $e) {
            return $this->sendError($e->getErrorMessage(), $e->getErrorCode());
        } catch (\Exception $e) {
            return $this->sendError($e->getMessage(), $e->getCode());
        }

    }

    /**
     * Retorna un token de acceso
     *
     * @param User $user
     * @return JsonResponse
     */
    protected function respondWithToken(User $user): JsonResponse
    {
        $user->tokens()->where('tokenable_id', $user->id)->delete();
        $expires_at = now()->addMinutes(config('sanctum.expiration'));
        $roles = $user->roles->pluck('name')->toArray();
        $token_name = $user->email.'|'.now();

        $response = array(
            'type' => 'Bearer',
            'token' => $user->createToken($token_name, $roles, $expires_at)->plainTextToken,
            'expires_at' => $expires_at,
            'user' => new UserResource($user)
        );

        return $this->sendResponse($response);
    }
}
