<?php

namespace ILC\AdminUsuarios\Imports;

use ILC\AdminUsuarios\Events\UsuarioCreado;
use ILC\AdminUsuarios\Events\UsuarioEditado;
use App\Models\User;
use ILC\AdminUsuarios\Models\ILCUserImports;
use ILC\AdminUsuarios\Rules\Curp as CurpRule;
use ILC\AdminUsuarios\Traits\ProgressTrackable;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithUpserts;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Validators\Failure;
class UsuariosImport implements
    ToModel,
    WithValidation,
    WithHeadingRow,
    SkipsEmptyRows,
    WithBatchInserts,
    WithUpserts,
    WithChunkReading,
    SkipsOnFailure,
    SkipsOnError,
    WithEvents
{
    use Importable, SkipsErrors, SkipsFailures, ProgressTrackable;

    protected array $importDetails = [
        'file_name' => '',
        'total_records_in_file' => 0,
        'records_added' => 0,
        'records_updated' => 0,
        'records_with_errors' => 0,
        'errors_detail' => null,
    ];

    protected int $successfulRegistrations = 0;
    protected int $failedRegistrations = 0;

    public function __construct($filename)
    {
        $this->importDetails['file_name'] = $filename;
    }

    /**
     * @param array $row
     * @return User|null
     */
    public function model(array $row): ?User
    {
        try {
            $strPassword = Str::random(10);
            $user = User::updateOrCreate(
                ['email' => $row['email']],
                [
                    'nombre' => $row['nombre'] ?? null,
                    'password' => Hash::make($strPassword),
                    'primer_apellido' => $row['primer_apellido'] ?? null,
                    'segundo_apellido' => $row['segundo_apellido'] ?? null,
                    'curp' => $row['curp'] ?? null,
                    'certificado' => $row['certificado'] ?? null,
                    'serie' => $row['serie'] ?? null,
                    'estado' => $row['estado'] ?? true,
                    'autenticacion' => $row['autenticacion'] ?? 'default',
                ]
            );

            $user->assignRole(Config('adminusuarios.default_role'));

            if ($user->wasRecentlyCreated) {
                Log::info('User created: ', [$user->email]);
                UsuarioCreado::dispatch($user->makeHidden('certificado'), $strPassword);
                $this->incrementAddedRows();
                $this->successfulRegistrations++;

            } else {
                Log::info('User updated: ', [$user->email]);
                UsuarioEditado::dispatch($user);
                $this->incrementUpdatedRows();
            }

            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            return $user;
        } catch (\Exception $e) {
            $this->importDetails['records_with_errors']++;
            $this->failedRegistrations++;

            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            return null;
        }

    }

    /**
     * Prepara los datos para la validación
     *
     * @param $data
     * @param $index
     * @return mixed
     */
    public function prepareForValidation($data, $index): mixed
    {

        $data['email'] = Str::lower($data['email'] ?? '');
        $data['curp_rule'] = $data['curp'] ?? '';
        $data['roles'] = [config('adminusuarios.default_role')];
        $data['estado'] = $data['estado'] ?? true;

        return $data;
    }

    /**
     * @param \Throwable $e
     * @return void
     */
    public function onError(\Throwable $e): void
    {
        $this->importDetails['records_with_errors']++;
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        return [
            'nombre' => 'required|string|max:255',
            'email' => 'sometimes|email|max:255',
            'password' => 'sometimes|max:10',
            'roles' => 'required|array|distinct|exists:roles,name',
            'permisos' => 'sometimes|array|distinct|exists:permissions,name',
            'serie' => 'required_if:autenticacion,=,certificado|max:60',
            'primer_apellido' => 'sometimes|string|max:100',
            'segundo_apellido' => 'sometimes|string|max:100',
            'curp' => 'sometimes|min:18|max:18',
            'curp_rule' => ['required', new CurpRule],
            'autenticacion' => Rule::in(['certificado', 'ldap', 'api', 'default'])
        ];
    }

    /**
     * @return array
     */
    public function customValidationMessages(): array
    {
        return trans('adminusuarios::validation.custom.excel');
    }

    /**
     * @return array
     */
    public function customValidationAttributes(): array
    {
        return trans('adminusuarios::validation.attributes');
    }


    /**
     * @return int
     */
    public function batchSize(): int
    {
        return 100;
    }


    /**
     * @return string
     */
    public function uniqueBy(): string
    {
        return 'email';
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 100;
    }


    /**
     * Retorna las filas que no pudieron importarse por algún error
     *
     * @param Failure ...$failures
     * @return void
     */
    public function onFailure(Failure ...$failures): void
    {
        $errorsDetail = [];

        foreach ($failures as $failure) {

            $this->importDetails['records_with_errors']++;
            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            foreach ($failure->errors() as $error) {
                if ($failure->attribute() !== 'curp_rule') {
                    $errorsDetail[$failure->row()][] = [
                        'campo' => $failure->attribute(),
                        'error' => $error,
                    ];
                }
            }
        }

        foreach ($errorsDetail as $key => $item) {
            $this->failures[$key] = [
                'fila' => $key,
                'errores' => $item,
            ];
        }

        $this->importDetails['errors_detail'] = json_encode(array_values($this->failures));

    }


    /**
     * @return array
     */
    public function getFailures(): array
    {
        return $this->failures;
    }

    /**
     * @return array
     */
    public function getImportDetails(): array
    {
        return $this->importDetails;
    }

    /**
     * @return void
     */
    public function storeImportDetails(): void
    {
        ILCUserImports::create($this->importDetails);
    }

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function(BeforeImport $event) {
                $totalRows = $event->getReader()->getTotalRows();
                $this->importDetails['total_records_in_file'] = array_sum($totalRows);

                $this->initializeUploadProgress($this->importDetails['total_records_in_file']);
            },
        ];
    }

    /**
     * @return void
     */
    protected function incrementAddedRows(): void
    {
        $this->importDetails['records_added']++;
    }

    /**
     * @return void
     */
    protected function incrementUpdatedRows(): void
    {
        $this->importDetails['records_updated']++;
    }

    public function getCertificateImportProgress(): \Illuminate\Http\JsonResponse
    {
        $progress = $this->getUploadProgress();
        Log::info('avance: ', [$progress]);
        return response()->json($progress);
    }
}
