<?php

namespace ILC\AdminUsuarios;

use ILC\AdminUsuarios\Commands\AdminUsuariosCommand;
use ILC\AdminUsuarios\Models\Perfil;
use ILC\AdminUsuarios\Providers\EventServiceProvider;
use ILC\AdminUsuarios\Providers\ILCAuthServiceProvider;
use Illuminate\Support\Facades\Auth;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Pluralizer;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class AdminUsuariosServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        $package
            ->name('adminusuarios')
            ->hasRoutes('api')
            ->hasConfigFile(['adminusuarios','audit', 'permission', 'sanctum', 'auth', 'excel'])
            ->hasViews()
            ->hasTranslations()
            ->hasMigrations([
                'create_audits_table',
                'create_permission_tables',
                'create_personal_access_tokens_table',
                'create_generos_table',
                'create_perfiles_table',
                'create_user_imports_table',
                ])
            ->publishesServiceProvider(AdminUsuariosServiceProvider::class)
            ->hasInstallCommand(function(InstallCommand $command) {
                $command
                    ->startWith(function(InstallCommand $command) {
                        $command->info('instalación del paquete ILC\AdminUsuarios!');
                    })
                    ->publishConfigFile()
                    ->publishMigrations()
                   ->askToRunMigrations()
                    ->endWith(function(InstallCommand $command) {
                        $command->info('corriendo seeders...');
                        Artisan::call('db:seed', ['--class' => 'ILC\\AdminUsuarios\\Database\\Seeders\\PermissionsSeeder']);
                        Artisan::call('db:seed', ['--class' => 'ILC\\AdminUsuarios\\Database\\Seeders\\GeneroSeeder']);
                        $command->info(Artisan::output());
                        App::currentLocale('es');
                        Pluralizer::useLanguage('spanish');
                        $command->info('finaliza instalación!');
                    });
            });
    }

    public function bootingPackage(): void
    {
        $this->publishes([
            base_path('vendor/maatwebsite/excel/config/excel.php') => config_path('excel.php'),
        ], 'excel.php');

        // Se declara estáticamente la relación del modelo User con el modelo Perfil para que no se altere la clase User
        \App\Models\User::resolveRelationUsing('perfil', function ($user) {
            return $user->hasOne(Perfil::class, 'user_id', 'id');
        });
    }

    public function packageRegistered(): void
    {
        $this->app->singleton('auth.driver', function ($app) {
            return new ILCAuthServiceProvider($app);
        });

        $this->app->register(EventServiceProvider::class);
        $this->app->register(ILCAuthServiceProvider::class);
    }

    public function packageBooted()
    {
        // Registramos las rutas del paquete al final cuando ya se botó el paquete.
        // Esto debería asegurar que se sobreescriban las rutas del paquete base siempre
        // y cuando se llamen igual y tengan la misma firma.
        $this->app->booted(function () {
            $this->registerCustomRoutes();
            if (!$this->app->environment('production')) {
                $faker = fake();
                $faker->addProvider(new \Database\Providers\PersonaProvider($faker));
            }
        });
    }

    protected function registerCustomRoutes()
    {
        Route::middleware('api')->group(__DIR__.'/../routes/api.php');
    }

}
