<?php

namespace ILC\AdminUsuarios\Auth\Strategies;

use ILC\AdminUsuarios\Auth\Contracts\AuthStrategyInterface;
use ILC\AdminUsuarios\Exceptions\InvalidCredentialsException;
use ILC\AdminUsuarios\Exceptions\UserNotFoundException;
use ILC\AdminUsuarios\Http\Requests\ValidateCredentialsRequest;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use ILC\AdminUsuarios\Helpers\CertHelper;
use ILC\AdminUsuarios\Exceptions\CertificateReadException;
use App\Models\User;

class CertificadoAuthStrategy implements AuthStrategyInterface {
    /**
     * @param ValidateCredentialsRequest $request
     * @return ?Authenticatable
     * @throws InvalidCredentialsException
     */
    public function validateCredentials(ValidateCredentialsRequest $request): ?Authenticatable
    {
        if (!Auth::loginUsingId($this->processCertificateFile($request)->id)) {
            throw new InvalidCredentialsException();
        }

        return Auth::user();
    }

    /**
     * @return int
     * @throws UserNotFoundException
     */
    public function revoke(): int
    {
        $user = Auth::user();

        if (!$user) {
            throw new UserNotFoundException();
        }

        return $user->tokens()->delete();
    }

    /**
     * Procesa el archivo del certificado
     *
     * @param Request $request
     * @return App\Models\User
     * @throws CertificateReadException
     */
    protected function processCertificateFile(ValidateCredentialsRequest $request)
    {
        if ($request->hasFile('certificado')) {
            // $mime = $request->certificado->getClientMimeType();
            // if($mime != 'application/pkix-cert'){
            //     throw new CertificateReadException('El campo certificado debe ser un archivo de tipo: cer', 422);
            // }

            $cert_helper = new CertHelper($request->certificado);
            $datos = $cert_helper->get_datos();
            if (count($datos)) {
                $user = User::where('email',$datos['email'])->first();
                if(!$user){
                    throw new CertificateReadException('Información incorrecta', 422);
                }elseif($this->validateSignature($request->firma, csrf_token(), $user)){
                    return $user;
                }
            }else {
                throw new CertificateReadException();
            }
        }
    }

    /**
     * Determina si la firma dada proviene del mensaje dado, según
     * el usuario dado (i.e. según su certificado público).
     *
     * @param
     * @param
     * @param  App\Models\User  $user
     * @return bool
     */
    private function validateSignature(string $signature64, string $message, User $user) {

        // Se convierte la firma, que viene en base 64,
        // a binario, pues openssl_verify así lo requiere.
        $signature = base64_decode($signature64);

        // Certificado PEM
        $certificadoPEM = $user->perfil->certificado;

        // Se verifica la firma.
        $resultado = openssl_verify($message, $signature, $certificadoPEM, OPENSSL_ALGO_SHA256);
        // $resultado = 1;

        // openssl_verify returns 1 if the signature is correct, 0 if it is incorrect, and -1 or false on error.
        // https://www.php.net/manual/en/function.openssl-verify.php
        switch ($resultado) {
            case 1:
                return true;
            case 0:
                throw new CertificateReadException('Error en la firma del mensaje', 422);
            case -1:
                throw new CertificateReadException('Error en la verificación de la firma', 422);
        }
    }

}

