<?php

namespace ILC\AdminUsuarios\Imports;

use App\Models\User;
use ILC\AdminUsuarios\Events\UsuarioCreado;
use ILC\AdminUsuarios\Events\UsuarioEditado;
use ILC\AdminUsuarios\Models\ILCUserImports;
use ILC\AdminUsuarios\Models\Perfil;
use ILC\AdminUsuarios\Rules\Curp as CurpRule;
use ILC\AdminUsuarios\Traits\ProgressTrackable;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithUpserts;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Validators\Failure;
class UsuariosImport implements
    ToModel,
    WithValidation,
    WithHeadingRow,
    SkipsEmptyRows,
    WithBatchInserts,
    WithUpserts,
    WithChunkReading,
    SkipsOnFailure,
    SkipsOnError,
    WithEvents
{
    use Importable, SkipsErrors, SkipsFailures, ProgressTrackable;

    protected array $importDetails = [
        'file_name' => '',
        'total_records_in_file' => 0,
        'records_added' => 0,
        'records_updated' => 0,
        'records_with_errors' => 0,
        'errors_detail' => null,
    ];

    protected int $successfulRegistrations = 0;
    protected int $successfulUpdates = 0;
    protected int $failedRegistrations = 0;

    public function __construct($filename)
    {
        $this->importDetails['file_name'] = $filename;
    }

    /**
     * @param array $row
     * @return User|null
     */
    public function model(array $row): ?User
    {
        try {
            // Determina si el usuario ya existía o no.
            // Si sí, lo almacena en la variable...
            $yaExistia = Perfil::where('curp', $row['curp'])->first()?->user;

            // Datos que se le asignarán al usuario.
            $datosAsignacion = [
                'email' => $row['email'] ?? null,
                'nombre' => $row['nombre'] ?? null,
                'primer_apellido' => $row['primer_apellido'] ?? null,
                'segundo_apellido' => $row['segundo_apellido'] ?? null,
                'curp' => $row['curp'] ?? null,
                'certificado' => $row['certificado'] ?? null,
                'serie' => $row['serie'] ?? null,
                'estado' => $row['estado'] ?? true,
                'autenticacion' => $row['autenticacion'] ?? 'default',
            ];

            // Si el usuario es nuevo, se le asigna una contraseña aleatoria.
            $strPassword = Str::random(10);
            if (! $yaExistia) {
                $datosAsignacion['password'] = Hash::make($strPassword);
            }

            // Creación/Actualización del usuario.
            // ¿Por qué "tap()"? Para obtener el Model después de actualizarlo: https://stackoverflow.com/a/60113718
            // ¿Por qué no "updateOrCreate"? Porque se busca mediante CURP (en el modelo Perfil), y no mediante email (en el modelo User).
            $user = $yaExistia ? tap($yaExistia)->update($datosAsignacion) : User::create($datosAsignacion);

            // Creación/Actualización del perfil asociado al usuario.
            if ($yaExistia) {
                $user->perfil->update($datosAsignacion);
            } else {
                $user->perfil()->create($datosAsignacion);
            }

            // Asignación de los roles especificados.
            $user->assignRole($row['roles']);

            if ($user->wasRecentlyCreated) {
                Log::info('User created: ', [$user->email, $user->perfil->curp]);
                UsuarioCreado::dispatch($user->makeHidden('certificado'), $strPassword);
                $this->incrementAddedRows();
                $this->successfulRegistrations++;

            } else {
                Log::info('User updated: ', [$user->email, $user->perfil->curp]);
                UsuarioEditado::dispatch($user);
                $this->incrementUpdatedRows();
                $this->successfulUpdates++;
            }

            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->successfulUpdates, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            return $user;
        } catch (\Exception $e) {
            $this->importDetails['records_with_errors']++;
            $this->failedRegistrations++;

            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->successfulUpdates, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            return null;
        }

    }

    /**
     * Prepara los datos para la validación
     *
     * @param $data
     * @param $index
     * @return mixed
     */
    public function prepareForValidation($data, $index): mixed
    {
        $data['email'] = Str::lower($data['email'] ?? '');
        $data['curp'] = Str::upper($data['curp']) ?? '';
        $data['roles'] = $data['roles'] ? explode(',', $data['roles']) : null;
        $data['estado'] = $data['estado'] ?? true;

        return $data;
    }

    /**
     * @param \Throwable $e
     * @return void
     */
    public function onError(\Throwable $e): void
    {
        // $this->importDetails['records_with_errors']++;
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        return [
            'nombre' => 'required|string|max:255',
            'email' => 'sometimes|email|max:255',
            'password' => 'sometimes|max:10',
            'roles' => 'required|array|distinct',
            'roles.*' => 'exists:roles,name',
            'permisos' => 'sometimes|array|distinct|exists:permissions,name',
            'serie' => 'required_if:autenticacion,=,certificado|max:60',
            'primer_apellido' => 'sometimes|string|max:100',
            'segundo_apellido' => 'sometimes|string|max:100',
            'curp' => ['required', 'min:18', 'max:18', new CurpRule],
            'autenticacion' => Rule::in(['certificado', 'default'])
        ];
    }

    /**
     * @return array
     */
    public function customValidationMessages(): array
    {
        return trans('adminusuarios::validation.custom.excel');
    }

    /**
     * @return array
     */
    public function customValidationAttributes(): array
    {
        return trans('adminusuarios::validation.attributes');
    }


    /**
     * @return int
     */
    public function batchSize(): int
    {
        return 100;
    }


    /**
     * @return string
     */
    public function uniqueBy(): string
    {
        return 'email';
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 100;
    }


    /**
     * Retorna las filas que no pudieron importarse por algún error
     *
     * @param Failure ...$failures
     * @return void
     */
    public function onFailure(Failure ...$failures): void
    {
        $errorsDetail = [];

        // Arreglo con los índices de los renglones que ya fueron
        // considerados para llevar el conteo de renglones con error.
        $countedRows = [];

        foreach ($failures as $failure) {

            // Conteo total de renglones con error.
            if (!in_array($failure->row(), $countedRows)) {
                $this->importDetails['records_with_errors']++;
                array_push($countedRows, $failure->row());
            }

            $processedFiles = $this->importDetails['records_added'] + $this->importDetails['records_updated'];
            $this->updateUploadProgress($this->successfulRegistrations, $this->successfulUpdates, $this->importDetails['records_with_errors'], $processedFiles);
            $this->getCertificateImportProgress();

            foreach ($failure->errors() as $error) {
                if (str_starts_with($failure->attribute(), 'roles')) {
                    $rolErrorArray = explode('.', $failure->attribute());
                    $rolErrorPos = end($rolErrorArray);
                    $errorsDetail[$failure->row()][] = [
                        'campo' => 'roles',
                        'error' => "Rol '".$failure->values()['roles'][$rolErrorPos]."' inválido. Verificar que el nombre sea correcto, y que no haya espacios de sobra.",
                    ];
                } else {
                    $errorsDetail[$failure->row()][] = [
                        'campo' => $failure->attribute(),
                        'error' => $error,
                    ];
                }
            }
        }

        foreach ($errorsDetail as $key => $item) {
            $this->failures[$key] = [
                'fila' => $key,
                'errores' => $item,
            ];
        }

        $this->importDetails['errors_detail'] = json_encode(array_values($this->failures));

    }


    /**
     * @return array
     */
    public function getFailures(): array
    {
        return $this->failures;
    }

    /**
     * @return array
     */
    public function getImportDetails(): array
    {
        return $this->importDetails;
    }

    /**
     * @return void
     */
    public function storeImportDetails(): void
    {
        ILCUserImports::create($this->importDetails);
    }

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function(BeforeImport $event) {
                $totalRows = $event->getReader()->getTotalRows();
                $this->importDetails['total_records_in_file'] = array_sum($totalRows);

                $this->initializeUploadProgress($this->importDetails['total_records_in_file']);
            },
        ];
    }

    /**
     * @return void
     */
    protected function incrementAddedRows(): void
    {
        $this->importDetails['records_added']++;
    }

    /**
     * @return void
     */
    protected function incrementUpdatedRows(): void
    {
        $this->importDetails['records_updated']++;
    }

    public function getCertificateImportProgress(): \Illuminate\Http\JsonResponse
    {
        $progress = $this->getUploadProgress();
        Log::info('avance: ', [$progress]);
        return response()->json($progress);
    }
}
