<?php

namespace ILC\AdminUsuarios\Http\Controllers;

use Exception;
use ILC\AdminUsuarios\Events\UsuarioCreado;
use ILC\AdminUsuarios\Events\UsuarioEditado;
use ILC\AdminUsuarios\Events\UsuarioEliminado;
use ILC\AdminUsuarios\Exceptions\AuthorizationException;
use ILC\AdminUsuarios\Exceptions\UserNotFoundException;
use ILC\AdminUsuarios\Http\Requests\CreateUserRequest;
use ILC\AdminUsuarios\Http\Requests\UpdateUserMassiveRequest;
use ILC\AdminUsuarios\Http\Requests\UpdateUserRequest;
use ILC\AdminUsuarios\Http\Resources\UserResource;
use App\Models\User;
use ILC\AdminUsuarios\Models\Perfil;
use ILC\AdminUsuarios\Traits\RollablePermissionable;
use ILC\AdminUsuarios\Traits\UserQueryTrait;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\PersonalAccessToken;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    use UserQueryTrait, RollablePermissionable;

    public function index(Request $request, $json = true)
    {
        if (!$this->hasPermission('visualizar-usuarios')) {
            throw new AuthorizationException();
        }
        $usersQuery = $this->buildUsersQuery($request);
        if ($request->filled('role')) {
            $usersQuery->whereHas('roles', function ($query) use ($request) {
                $query->where('name', $request->role);
            });
        }
        if ($request->filled('status')) {
            $usersQuery->whereHas('perfil', function ($query) use ($request) {
                $query->where('estado',filter_var($request->status, FILTER_VALIDATE_BOOLEAN));
            });
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $usersQuery->whereHas('perfil', function ($query) use ($search) {
                $query->whereRaw("concat_ws(' ', nombre, primer_apellido, segundo_apellido) ilike ?", ["%$search%"]) // Buscar en la tabla perfils
                ->orWhere('curp', 'ilike', "%$search%");
            })->orWhere('email', 'ilike', "%$search%");
        }
        if(!$json){
            return $usersQuery;
        }

        $this->applySorting($usersQuery, $request);
        $perPage = $request->get('per_page', 10);
        $users = $usersQuery->paginate($perPage);

        return $this->sendResponse([
            'total' => $users->total(),
            'filter' => $users->total(),
            'pages' => $users->lastPage(),
            'data' => $users->items()
        ]);
    }

    public function store(CreateUserRequest $request)
    {
        try {
            $user = User::create($request->only(['email','password']));
            $user->perfil()->create($request->all());

            $this->assignRolesAndPermissions($user, $request);

            if ($request->enviar_correo) {
                UsuarioCreado::dispatch($user, $request->secret);
            }

            $response = $this->sendResponse(new UserResource($user), 201);
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }
        return $response;
    }

    /**
     * Actualiza el registro de un usuario
     *
     * @param Request $request
     * @param User $user
     * @return JsonResponse
     */
    public function show(Request $request, User $user): JsonResponse
    {
        try{
            if (!$this->hasPermission('visualizar-usuarios')) {
                throw new AuthorizationException();
            }

            if (!$user) {
                throw new UserNotFoundException();
            }

            $response = $this->sendResponse(new UserResource($user));
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }

    /**
     * Actualiza el registro de un usuario
     *
     * @param UpdateUserRequest $request
     * @param User $user
     * @return JsonResponse
     */
    public function update(UpdateUserRequest $request, User $user): JsonResponse
    {
        try {
            $user->update($request->only(['email']));
            $perfilData = array_filter($request->all(), fn($key) => in_array($key, $user->perfil->getFillable()),
                ARRAY_FILTER_USE_KEY
            );
            $user->perfil->update($perfilData);

            $this->assignRolesAndPermissions($user, $request);
            UsuarioEditado::dispatch($user->makeHidden('certificado'));

            $response = $this->sendResponse(new UserResource($user));
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }


    /**
     * Marca un usuario como eliminado (no lo elimina de la bd).
     *
     * @param Request $request
     * @param User $user
     * @return JsonResponse
     * @throws AuthorizationException
     */
    public function destroy(Request $request, User $user) : JsonResponse
    {
        if (!$this->hasPermission('eliminacion-de-usuarios')) {
            throw new AuthorizationException();
        }
        try {
            if(!$user){
                throw new UserNotFoundException();
            }
            $this->removeUsers($user);
            $response = $this->sendResponse(null, 204);

        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), $e->getCode());
        }

        return $response;
    }

    /**
     * Toggle the specified resource from storage.
     *
     * @param User $user
     * @return JsonResponse
     * @throws AuthorizationException
     */
    public function status(User $user)
    {
        if (!$this->hasPermission('edicion-de-usuarios')) {
            throw new AuthorizationException();
        }

        if(!$user){
            throw new UserNotFoundException();
        }

        try{
            $user->estado = !$user->estado;
            $user->save();

            $response = $this->sendResponse(null, 200);
        } catch (\Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }

    /**
     * Edicion masiva de usuario.
     *
     * @param UpdateUserMassiveRequest $request
     * @return JsonResponse
     */
    public function update_massive(UpdateUserMassiveRequest $request): JsonResponse
    {
        try {
            if($request->filled('users')){ //static
                foreach($request->users as $item){
                    $user = User::find($item['id']);
                    match ($request->action) {
                        'roles' => $user->syncRoles([$request->roles]),
                        'statusses' => $user->perfil()->update(['estado' => $request->statusses]),
                        'delete' => $this->removeUsers($user)
                    };
                }
            }else{ //massive
                $users = $this->index(request(), false)->pluck('users.id');
                switch($request->action){
                    case('statusses'):
                        if($request->statusses == 0){
                            PersonalAccessToken::whereIn('tokenable_id', $users)->delete();
                        }
                        User::whereIn('id', $users)->perfil()->update(['estado' => $request->statusses]);
                        break;
                    case('delete'):
                        PersonalAccessToken::whereIn('tokenable_id', $users)->delete();
                        \DB::table('model_has_permissions')->whereIn('model_id', $users)->delete();
                        \DB::table('model_has_roles')->whereIn('model_id', $users)->delete();
                        User::whereIn('id', $users)->perfil()->delete();
                        User::whereIn('id', $users)->delete();
                        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
                        break;
                    case('roles'):
                        $role = Role::where('name', $request->roles)->first();
                        \DB::table('model_has_roles')->whereIn('model_id', $users)->update(['role_id' => $role->id]);
                        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
                        break;
                }
            }

            $response = $this->sendResponse(null, 200);
        } catch (\Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }

    /**
     * Remueve usuarios, asi como roles, permisos y tokens de acceso
     *
     * @param User $user
     * @return void
     */
    protected function removeUsers(User $user): void
    {
        $user->syncPermissions();
        $user->syncRoles();
        $user->delete();
        $user->tokens()->delete();
        UsuarioEliminado::dispatch($user->makeHidden('certificado'));
    }
}
