<?php

namespace ILC\AdminUsuarios\Http\Controllers;



use Exception;
use ILC\AdminUsuarios\Events\ArchivoUsuariosCargado;
use ILC\AdminUsuarios\Events\UsuarioCreado;
use ILC\AdminUsuarios\Events\UsuarioEditado;
use ILC\AdminUsuarios\Exceptions\AuthorizationException;
use ILC\AdminUsuarios\Exceptions\CertificateReadException;
use ILC\AdminUsuarios\Exceptions\UserNotFoundException;
use ILC\AdminUsuarios\Helpers\CertHelper;
use ILC\AdminUsuarios\Http\Requests\CreateUsersWithCertificatesRequest;
use ILC\AdminUsuarios\Http\Requests\ImportExcelUsersRequest;
use ILC\AdminUsuarios\Http\Resources\ImportResource;
use ILC\AdminUsuarios\Imports\UsuariosImport;
use App\Models\User;
use ILC\AdminUsuarios\Models\ILCUserImports;
use ILC\AdminUsuarios\Traits\HandlesImportDetails;
use ILC\AdminUsuarios\Traits\ProgressTrackable;
use ILC\AdminUsuarios\Traits\RollablePermissionable;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Validators\ValidationException;

class ImportController extends Controller
{
    use ProgressTrackable, HandlesImportDetails, RollablePermissionable;

    /**
     * Retorna el listado de archivos importados paginado
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $perPage = 10;
        $imports = ILCUserImports::paginate($perPage);
        $data = ImportResource::collection($imports)->response()->getData(true);

        return $this->sendResponse($data);
    }

    public function show(Request $request, $import): JsonResponse
    {
        try {
            if (!$this->hasPermission('visualizar-usuarios')) {
                throw new AuthorizationException();
            }

            $import = ILCUserImports::find($import);

            if (!$import) {
                throw new UserNotFoundException();
            }

            $response = $this->sendResponse(new ImportResource($import, 'show'));
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }


    /**
     * Crea usuarios a partir de archivos .cer subidos
     *
     * @param CreateUsersWithCertificatesRequest $request
     * @return JsonResponse
     */
    public function certificate(CreateUsersWithCertificatesRequest $request): JsonResponse
    {
        $successfulRegistrations = 0;
        $successfulUpdates = 0;
        $failedRegistrations = 0;
        $certificates = $request->file('certificados');
        $totalFiles = count($certificates);
        $failedRegistrationsDetail = ['errors_detail' => []];

        $this->initializeUploadProgress($totalFiles);

        foreach ($certificates as $index => $certificate) {
            $importDetails = $this->initializeImportDetails($certificate->getClientOriginalName());

            if (!$this->certificateIsValidAndReadable($certificate)) {
                $this->handleCertificateError($certificate, $importDetails, $failedRegistrationsDetail, 'El certificado no cuenta con los datos requeridos o no son válidos. Err101');
                $failedRegistrations++;
                continue;
            }

            DB::beginTransaction();

            try {
                $res = $this->createOrUpdateUserFromCertificateFile($certificate, $request);
                if (!$res['user']) {
                    $this->handleCertificateError($certificate, $importDetails, $failedRegistrationsDetail, 'El certificado no cuenta con los datos requeridos o no son válidos. Err101');
                    $failedRegistrations++;
                    DB::rollBack();
                    continue;
                }

                $this->addRecordToImportDetails($importDetails);
                ILCUserImports::create($importDetails);

                if ($res['action'] === 'created') {
                    $successfulRegistrations++;
                } else if ($res['action'] === 'updated') {
                    $successfulUpdates++;
                }

                DB::commit();

            } catch (Exception $e) {
                DB::rollBack();
                Log::error('Error processing certificate: ' . $e->getMessage(), [
                    'line' => $e->getLine(),
                    'file' => $e->getFile(),
                    'trace' => $e->getTraceAsString()
                ]);

                $this->handleCertificateError($certificate, $importDetails, $failedRegistrationsDetail, 'El certificado no cuenta con los datos requeridos o no son válidos. Err102.');
                $failedRegistrations++;
                continue;
            }

            $this->updateUploadProgress($successfulRegistrations, $successfulUpdates, $failedRegistrations, $index + 1);
        }

        $importResponse = $this->getCertificateImportProgress();
        $importResults = json_decode($importResponse->getContent(), true);

        $statusMessage = count($failedRegistrationsDetail['errors_detail']) ?
            'Se completó la importación con los siguientes errores.' :
            'Se completó la importación con éxito.';

        $response = $this->sendResponse([
            'status' => $statusMessage,
            'total_records' => $importResults['total_records'],
            'records_added' => $importResults['successful_registrations'],
            'records_updated' => $importResults['successful_updates'],
            'records_with_errors' => $importResults['failed_registrations'],
            'errors_detail' => $failedRegistrationsDetail['errors_detail'],
        ], 201);

        $this->registerFailedImportsInDatabase($failedRegistrationsDetail);
        $this->clearUploadProgress();

        return $response;
    }

    private function handleCertificateError($certificate, &$importDetails, &$failedRegistrationsDetail, $errorMessage)
    {
        $this->addErrorToImportDetails($importDetails, $certificate->getClientOriginalName(), $errorMessage);
        $this->recordFailedRegistrationDetail($failedRegistrationsDetail, $certificate->getClientOriginalName(), $errorMessage);
    }

    public function getCertificateImportProgress()
    {
        $progress = $this->getUploadProgress();
        Log::info('avance: ', [$progress]);
        return response()->json($progress);
    }


    /**
     * Revisa que el archivo subido sea un certificado válido y se pueda leer.
     *
     * @param UploadedFile $file
     * @return bool
     */
    protected function certificateIsValidAndReadable(UploadedFile $file): bool
    {
        $certHelper = new CertHelper($file);
        $data = $certHelper->get_datos();
        $mimeType = $file->getClientMimeType();

//        if ($mimeType === 'application/pkix-cert' && count($data) ) {
//            return true;
//        }

        return true;
    }

    /**
     * Importa los usuarios a partir de una plantilla de Excel
     *
     * @param ImportExcelUsersRequest $request
     * @return JsonResponse
     */
    public function excel(ImportExcelUsersRequest $request): JsonResponse
    {
        $filePath = null;

        try {
            $file = $request->file('usuarios');
            $filePath = $file->store('uploads');
            $filename = $file->getClientOriginalName();
            event(new ArchivoUsuariosCargado($file));
            $import = new UsuariosImport($filename);
            $import->import(storage_path('app/' . $filePath));
            Log::info(['import', $import]);
            $import->storeImportDetails();
            $failures = $import->getFailures();
            $importDetails = $import->getImportDetails();

            $response = $this->sendResponse([
                'status' => count($failures) ? 'Se completó la importación con los siguientes errores.' : 'Se completó la importación con éxito.',
                'total_records' => $importDetails['total_records_in_file'] -1,
                'records_added' => $importDetails['records_added'],
                'records_updated' => $importDetails['records_updated'],
                'records_with_errors' => $importDetails['records_with_errors'],
                'errors_detail' => count($failures) ? array_values($failures) : [],
            ]);
        } catch (ValidationException $e) {
            $failures = $e->failures();
            $result = [];

            foreach ($failures as $failure) {
                $result[] = [
                    'fila' => $failure->row(),
                    'campo' => $failure->attribute(),
                    'error' => $failure->errors(),
                ];
            }

            $response = $this->sendError($result, 422);
        }
        catch (\Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        } finally {

            if ($filePath) {
                Storage::delete($filePath);
            }
        }

        return $response;
    }

    protected function createOrUpdateUserFromCertificateFile(UploadedFile $certificate, $request): array
    {
        $certHelper = new CertHelper($certificate);
        $certData = $certHelper->get_datos();

        if (!count($certData)) {
            throw new CertificateReadException('No se encontraron datos válidos en el certificado.');
        }

        if (!$this->requiredFieldsForCertificateArePresent($certData)) {
            throw new CertificateReadException('No se encontraron los datos requeridos en el archivo.');
        }

        $processedData = $this->prepareDataForUserCreation($certData);
        $secret = $processedData['password'];
        $processedData['password'] = Hash::make($secret);
        $processedData['autenticacion'] = 'certificado';

        // Intentar encontrar un usuario existente basado en la información del perfil (curp)
        $user = User::whereHas('perfil', function ($q) use ($processedData) {
            $q->where('curp', 'ilike', $processedData['curp']);
        })->first();

        // Si no existe un usuario, crear uno nuevo
        if (!$user) {
            $user = User::create($processedData);
            $user->perfil()->create($processedData);

            $this->assignDefaultRoleWhenMassCreated($user);
            $this->assignRolesAndPermissions($user, $request);

            UsuarioCreado::dispatch($user, $secret);

            return [
                'user' => $user,
                'action' => 'created',
            ];
        }

        // Actualizar el usuario existente
        $perfilData = array_filter($processedData, fn($key) => in_array($key, $user->perfil->getFillable()),
            ARRAY_FILTER_USE_KEY
        );
        $user->perfil()->update($perfilData);

        UsuarioEditado::dispatch($user);

        return [
            'user' => $user,
            'action' => 'updated',
        ];
    }


    /**
     * Revisa que los campos requeridos para registrar un usuario usando certificado estén presentes.
     *
     * @param array $certData - Los datos obtenidos después de procesar el certificado.
     * @return bool
     */
    protected function requiredFieldsForCertificateArePresent(array $certData): bool
    {
        $requiredFields = ['nombre', 'email', 'curp', 'serie'];

        if (!Arr::has($certData, $requiredFields)) {
            return false;
        }

        return true;
    }


    /**
     * Asigna el rol por default para los usuarios importados (específicado en el archivo de config. "adminusuarios")
     *
     * @param App\Models\User $user
     * @return void
     */
    protected function assignDefaultRoleWhenMassCreated(User $user): void
    {
        $user->assignRole(Config('adminusuarios.default_role'));
    }


    /**
     * Prepara los datos obtenidos del certificado para crear el usuario.
     *
     * @param array $certData
     * @return array
     */
    protected function prepareDataForUserCreation(array $certData): array
    {
        $certData['email'] = Str::lower($certData['email']) ?? '';
        $certData['curp'] = Str::upper($certData['curp']) ?? '';
        $certData['curp_rule'] = $certData['curp'] ?? '';
        $certData['password'] = Str::random(10);

        return $certData;
    }

    protected function registerFailedImportsInDatabase(array $failedImports): void
    {
        foreach ($failedImports['errors_detail'] as $failed) {

            ILCUserImports::create([
                'file_name' => $failed['campo'],
                'total_records_in_file' => 1,
                'records_added' => 0,
                'records_updated' => 0,
                'records_with_errors' => 1,
                'errors_detail' => json_encode([
                    ["error" => $failed['error']]
                ])
            ]);
        }
    }
}
