<?php

namespace ILC\AdminUsuarios\Http\Controllers;



use Exception;
use ILC\AdminUsuarios\Events\UsuarioCreado;
use ILC\AdminUsuarios\Events\UsuarioEditado;
use ILC\AdminUsuarios\Events\UsuarioEliminado;
use ILC\AdminUsuarios\Events\ContrasenaTemporal;
use ILC\AdminUsuarios\Exceptions\AuthorizationException;
use ILC\AdminUsuarios\Exceptions\UserNotFoundException;
use ILC\AdminUsuarios\Exceptions\UnexpectedServerException;
use ILC\AdminUsuarios\Http\Requests\CreateUserRequest;
use ILC\AdminUsuarios\Http\Requests\UpdateUserRequest;
use ILC\AdminUsuarios\Http\Requests\UpdateUserMassiveRequest;
use ILC\AdminUsuarios\Http\Resources\UserResource;
use ILC\AdminUsuarios\Http\Resources\UserCollection;
use ILC\AdminUsuarios\Http\Resources\UserFilterCollection;
use ILC\AdminUsuarios\Models\ILCUser;
use ILC\AdminUsuarios\Traits\UserQueryTrait;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{

    use UserQueryTrait;

    public function index(Request $request): JsonResponse
    {
        if (!$this->hasPermission('visualizar-usuarios')) {
            throw new AuthorizationException();
        }

        $usersQuery = $this->buildUsersQuery($request);
        if($request->filled('role')){
            $usersQuery->where('roles.name', $request->role);
        }
        if($request->filled('status')){
            $usersQuery->where('estado', filter_var($request->status, FILTER_VALIDATE_BOOLEAN));
        }
        if($request->filled('search')){
            $search = $request->search;
            $usersQuery->where(function($query) use ($search) {
                        $query->where(\DB::raw("concat_ws(' ', nombre, primer_apellido, segundo_apellido)"), 'ilike', "%$search%")
                        ->orWhere('email', 'ilike', "%$search%")
                        ->orWhere('curp', 'ilike', "%$search%")
                        ->orWhere('serie', 'ilike', "%$search%");
            });
        }

        $usersQuery->get();

        $this->applySorting($usersQuery, $request);
        $perPage = $request->get('per_page', 10);
        $users = $usersQuery->paginate($perPage);

        return $this->sendResponse([
            'total' => $users->total(),
            'filter' => $users->total(),
            'pages' => $users->lastPage(),
            'data' => $users->items()
        ]);
    }


    /**
     * Registra un nuevo usuario
     *
     * @param CreateUserRequest $request
     * @return JsonResponse
     */
    public function store(CreateUserRequest $request): JsonResponse
    {
        try {
            $user = ILCUser::updateOrCreate(
                ['email' => $request->email],
                $request->except('secret')
            );
            $this->assignRolesAndPermissions($user, $request);

            if ($request->enviar_correo) {
                UsuarioCreado::dispatch($user, $request->secret);
            }

            $response = $this->sendResponse(new UserResource($user), 201);
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }
        return $response;
    }


    /**
     * Actualiza el registro de un usuario
     *
     * @param Request $request
     * @param ILCUser $user
     * @return JsonResponse
     */
    public function show(Request $request, ILCUser $user): JsonResponse
    {
        try{
            if (!$this->hasPermission('visualizar-usuarios')) {
                throw new AuthorizationException();
            }

            // $user = ILCUser::find($request->id);

            if (!$user) {
                throw new UserNotFoundException();
            }

            $response = $this->sendResponse(new UserResource($user));
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }


    /**
     * Actualiza el registro de un usuario
     *
     * @param UpdateUserRequest $request
     * @param ILCUser $user
     * @return JsonResponse
     */
    public function update(UpdateUserRequest $request, ILCUser $user): JsonResponse
    {
        try {
            $user = ILCUser::find($request->id);
            $user->update($request->all());

            $this->assignRolesAndPermissions($user, $request);
            UsuarioEditado::dispatch($user->makeHidden('certificado'));

            $response = $this->sendResponse(new UserResource($user));
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }

    /**
     * Marca un usuario como eliminado (no lo elimina de la bd).
     *
     * @param Request $request
     * @param ILCUser $user
     * @return JsonResponse
     * @throws AuthorizationException
     */
    public function destroy(Request $request, ILCUser $user) : JsonResponse
    {
        if (!$this->hasPermission('eliminacion-de-usuarios')) {
            throw new AuthorizationException();
        }

        $user = ILCUser::find($request->id);
        try {
            if(!$user){
                throw new UserNotFoundException();
            }

            $this->removeUsers($user);
            $response = $this->sendResponse(null);
        } catch (Exception $e) {
            $response = $this->sendError($e->getMessage(), $e->getCode());
        }

        return $response;
    }


    /**
     * Edicion masiva de usuario.
     *
     * @param UpdateUserMassiveRequest $request
     * @return JsonResponse
     */
    public function update_massive(UpdateUserMassiveRequest $request) : JsonResponse
    {
        try {
            foreach($request->users as $item){
                $user = ILCUser::find($item['id']);
                match ($request->action) {
                    'role' => $user->syncRoles([$request->role]),
                    'status' => $user->update(['estado' => $request->status]),
                    'delete' => $this->removeUsers($user)
                };
            }
            $response = $this->sendResponse(null, 200);
        } catch (\Exception $e) {
            $response = $this->sendError($e->getMessage(), 500);
        }

        return $response;
    }


    /**
     * Envia una contraseña temporal al usuario
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function reset(Request $request): JsonResponse
    {
        try{
            $user = ILCUser::where(['email' => $request->email])->first();
            if(!$user){
                throw new UserNotFoundException();
            }

            if(!$user->estado){
                throw new UnexpectedServerException('Usuario inactivo', 200);
            }

            $secret = Str::random(10);
            $user->update(['password' => Hash::make($secret)]);
            $response = $this->sendResponse(null, 200);
            ContrasenaTemporal::dispatch($user, $secret);
        } catch (\Exception $e) {
            $response = $this->sendError($e->getMessage(), $e->getCode());
        }

        return $response;
    }

    /**
     * Asigna roles y permisos al usuario
     *
     * @param ILCUser $user
     * @param Request $request
     * @return void
     */
    protected function assignRolesAndPermissions(ILCUser $user, Request $request): void
    {
        if($request->filled('roles')) {
            $user->syncRoles([$request->roles]);
        }

        if($request->filled('permisos')) {
            $user->syncPermissions([$request->permisos]);
        }
    }


    /**
     * Remueve usuarios, asi como roles, permisos y tokens de acceso
     *
     * @param ILCUser $user
     * @return void
     */
    protected function removeUsers(ILCUser $user): void
    {
        $user->syncPermissions();
        $user->syncRoles();
        $user->delete();
        $user->tokens()->delete();
        UsuarioEliminado::dispatch($user->makeHidden('certificado'));
    }

}
