<?php
namespace ILC\AdminUsuarios\Http\Requests;

use ILC\AdminUsuarios\Auth\AuthManagerContext;
use ILC\AdminUsuarios\Auth\Strategies\ApiAuthStrategy;
use ILC\AdminUsuarios\Auth\Strategies\CertificadoAuthStrategy;
use ILC\AdminUsuarios\Auth\Strategies\DefaultAuthStrategy;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Lang;

class ValidateCredentialsRequest extends FormRequest
{
    protected AuthManagerContext $authManager;

    public function __construct(AuthManagerContext $authManager)
    {
        $this->authManager = $authManager;
        parent::__construct();
    }

    /**
     * Determina si el usuario está autorizado para hacer la petición.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Lógica para obtener las reglas de validación
     * de acuerdo a la estrategía utilizada
     *
     * @return array
     */
    public function rules(): array
    {
        $strategy = $this->authManager->getStrategy();

        if ($strategy instanceof CertificadoAuthStrategy) {
            return $this->validateCertificadoCredentials();
        } elseif ($strategy instanceof ApiAuthStrategy || $strategy instanceof DefaultAuthStrategy) {
            return $this->validateAPIAndDefaultCredentials();
        }

        return [];
    }

    /**
     * Reglas de validación para inicio de sesión con certificado
     *
     * @return array
     */
    protected function validateCertificadoCredentials(): array
    {
        return [
            'certificado' => 'required|file',
            'firma' => 'required',
        ];
    }

    /**
     * Reglas de validación para inicio de sesión con formulario
     *
     * @return array
     */
    protected function validateAPIAndDefaultCredentials(): array
    {
        return [
            'email' => 'required|email',
            'password' => 'required|min:6',
        ];
    }

    /**
     * Las traducciones de las reglas
     *
     * @return array
     */
    public function messages(): array
    {
        return trans('adminusuarios::auth.custom');
    }

    /**
     * Las traducciones de los atributos a validar
     *
     * @return array
     */
    public function attributes(): array
    {
        return trans('adminusuarios::auth.attributes');
    }
}

