<?php

namespace ILC\AdminUsuarios\Http\Requests;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use ILC\AdminUsuarios\Helpers\CertHelper;
use ILC\AdminUsuarios\Rules\Curp as CurpRule;
use ILC\AdminUsuarios\Exceptions\CertificateReadException;
use Illuminate\Validation\Rule;

class CreateUserRequest extends FormRequest
{
    protected $stopOnFirstFailure = true;

    /**
     * Determina si el usuario está autorizado para hacer la petición.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return auth()->check() && auth()->user()->hasPermissionTo('creacion-de-usuarios');
    }

    /**
     * Procesa campos antes de validarlos
     *
     * @return void
     */
    protected function prepareForValidation(): void
    {
        $this->processCertificateFile($this);
        $this->merge([
            'email' => Str::lower($this->email),
            'curp_rule' => $this->curp,
            'enviar_correo' => filter_var($this->enviar_correo, FILTER_VALIDATE_BOOLEAN)
        ]);
    }

    /**
     * Handle a passed validation attempt.
     */
    protected function passedValidation(): void
    {
        $secret = (!$this->filled('password') ? Str::random(10) : $this->password);
        $this->merge(['password' => Hash::make($secret), 'secret' => $secret]);
        $this->request->remove('valid');
        $this->request->remove('curp_rule');
        $this->files->remove('certificado');
        $this->convertedFiles = null;
    }

    /**
     * Las reglas de validación para crear usuarios
     *
     * @return array
     */
    public function rules(): array
    {
        return [
            'nombre' => 'required|max:255',
            'primer_apellido' => 'nullable|string|max:100',
            'segundo_apellido' => 'nullable|string|max:100',
            'email' => ['required','email','max:255',Rule::unique('users', 'email')->ignore($this->id)],
            'curp' => ['required','min:18','max:18', Rule::unique('users', 'curp')->ignore($this->id),],
            'curp_rule' => ['required', new CurpRule],
            'password' => 'sometimes|max:15',
            'roles' => 'required|array|min:1|distinct|exists:roles,name',
            'permisos' => 'sometimes|array|distinct|exists:permissions,name',
            'serie' => 'required_if:autenticacion,=,certificado|max:60',
            'autenticacion' => 'required|in:certificado,ldap,api,default',
            'estado' => 'sometimes|boolean',
            'enviar_correo' => 'sometimes|boolean',
        ];
    }

    /**
     * Las traducciones de las reglas
     *
     * @return array
     */
    public function messages(): array
    {
       return trans("adminusuarios::validation");
    }

    /**
     * Las traducciones de los atributos a validar
     *
     * @return array
     */
    public function attributes(): array
    {
        return trans("adminusuarios::validation.attributes");
    }

    /**
     * Procesa el archivo del certificado
     *
     * @param Request $request
     * @return void
     */
    protected function processCertificateFile(Request $request): void
    {
        if ($request->hasFile('certificado')) {
            // $mime = $request->certificado->getClientMimeType();
            // if($mime != 'application/pkix-cert'){
            //     throw new CertificateReadException('El campo certificado debe ser un archivo de tipo: cer', 422);
            // }

            $cert_helper = new CertHelper($request->certificado);
            $datos = $cert_helper->get_datos();
            if (count($datos)) {
                $current = date('Y-m-d');
                if($datos['valid'] < $current){
                    throw new CertificateReadException('El certificado no es vigente');
                }
                $request->merge(['autenticacion' => 'certificado']);

                foreach($datos as $key => $value){
                    $request->merge([$key => $value]);
                }
            }else {
                throw new CertificateReadException();
            }
        }
    }
}
