<?php

namespace ILC\AdminUsuarios\Helpers;

use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use ILC\AdminUsuarios\Models\ILCUser;

class CertHelper
{
    /**
     * Constructor.
     *
     * @param  $file Archivo con el certificado X509 que se manejará.
     * @param  $user Usuario al cual le pertenece el certificado.
     */
    public function __construct(UploadedFile $file, ILCUser $user = null)
    {
        $this->file = $file;
        $this->user = $user;
        $this->datos = $this->extraer_datos_legibles();
    }

    /**
     * Extracción de los datos del certificado público.
     *
     * @return
     */
    private function extraer_datos_legibles()
    {
        // Lectura del certificado.
        $certx509 = CertHelper::certx509_from_file($this->file);
        $certificado = $certx509;
        $certx509 = openssl_x509_parse($certx509);

        // Validación en caso de error de lectura
        if(!$certx509){
            return [];
        }

        // Incluye parte de los datos del certificado.
        $certx509_name = explode('/', $certx509['name']);

        // Arreglo con los datos a extraer.
        $datos = [
            'nombre' => explode('=', $certx509_name[1])[1],
            'primer_apellido' => "",
            'segundo_apellido' => "",
            'curp' => explode('=', $certx509_name[7])[1],
            // 'rfc' => explode('=', $certx509_name[6])[1],
            'email' => explode('=', $certx509_name[5])[1],
            'serie' => preg_replace('/.(.)/', '$1', $certx509['serialNumberHex']),
            'certificado' => $certificado,
            'valid' => date('Y-m-d', $certx509['validTo_time_t'])
        ];

        $datos = $this->fixName($datos);
        return $datos;
    }

    /**
     * Obtención de los datos del certificado público;
     * los datos "finales" del certificado, los datos legibles por los usuarios.
     *
     * @param  $solo_valores_nuevos Si es 'true', y el certificado contiene algún valor ya perteneciente al usuario, la función no regresa este valor.
     * @return
     */
    public function get_datos(bool $solo_valores_nuevos = false)
    {
        // Copia de los datos originales.
        $datos = $this->datos;

        // Si se tiene un usuario, y sólo se desean valores nuevos,
        // se quitan los valores repetidos/iguales que el usuario ya tenga.
        if ($this->user && $solo_valores_nuevos) {
            foreach ($datos as $key => $value) {
                if ($value == $this->user[$key])
                    unset($datos[$key]);
            }
        }

        // Se regresan los datos.
        return $datos;
    }

    /**
     * Almacena el certificado público dado.
     *
     * @param  $file Archivo con el certificado X509.
     * @param  \App\Models\User  $user  Usuario al cual le pertenece el certificado (sólo es necesario al actualizar los datos del usuario).
     * @return
     */
    public function almacenar_certificado()
    {
        // Dónde y cómo se guardará el certificado.
        $dir = $this->datos['curp'].'/';
        $filename = $this->datos['serie_certificado'].'.cer';

        // Datos (sin repetir) del certificado.
        $datos = $this->get_datos(true);

        // Posibles errores.
        $errores = [];
        if (Storage::disk('certificados')->exists($dir.$filename))
            $errores['certificado'] = 'Archivo ya guardado en almacenamiento.';
        if (!User::where('serie_certificado', $this->datos['serie_certificado'])->get()->isEmpty())
            $errores['serie_certificado'] = 'Serie de certificado ya registrado en BD.';
        if (isset($datos['email']) && !User::where('email', $datos['email'])->get()->isEmpty())
            $errores['email'] = 'Correo ya registrado en BD.';
        if ($errores)
            throw ValidationException::withMessages($errores);

        // Si lo anterior no falla, se guarda el certificado.
        Storage::disk('certificados')->putFileAs($dir, $this->file, $filename);
    }

    /**
     * Lectura de un certificado público.
     *
     * @param  $file Archivo con el certificado X509.
     * @return Arreglo con la información del certificado.
     */
    static function certx509_from_file(UploadedFile $file)
    {
        // Se lee el archivo (que viene en formato DER),
        // y se convierte a PEM.
        $certDER = file_get_contents($file);
        $certPEM = CertHelper::der_to_pem($certDER);

        // Lectura del certificado.
        // https://www.php.net/manual/es/function.openssl-x509-parse.php
        return $certPEM;
    }

    /**
     * Conversión del formato DER al formato PEM.
     *
     * @param  $inDER Cadena que representa el contenido de un archivo en formato DER.
     * @return Cadena que representa el contenido del archivo en formato PEM.
     */
    static function der_to_pem(string $inDER)
    {
        // https://stackoverflow.com/a/22743616
        $inPEM =
        '-----BEGIN CERTIFICATE-----'.PHP_EOL
        .chunk_split(base64_encode($inDER), 64, PHP_EOL)
        .'-----END CERTIFICATE-----'.PHP_EOL;

        return $inPEM;
    }

    private function fixName($data, $value = null, $count = 0) {
        $prep = ['DE', 'DEL', 'LA', 'LAS', 'LOS'];
        $name = explode(' ', $data['nombre']);
        if(count($name)) {
            foreach(array_reverse($name) as $item){
                if($count < 2){
                    if(in_array($item, $prep)){
                        $value = $item.' '.$value;
                    }else{
                        if(is_null($value)){
                            $value = $item;
                        }else{
                            $value = ($count == 1 ? $value : $item.'|'.$value);
                            $count++;
                        }
                    }
                    if($count < 2){ array_pop($name);}
                }
            }

            if(count($name)){
                $full_name = explode('|', $value.'|'.implode(' ', $name));
                [$data['primer_apellido'], $data['segundo_apellido'], $data['nombre']] = $full_name;
            }else{
                $full_name = explode('|', $value);
                [$data['nombre'], $data['primer_apellido']] = $full_name;
            }
        }

        return $data;
    }

}
