<?php

namespace ILC\AdminUsuarios;

use ILC\AdminUsuarios\Commands\AdminUsuariosCommand;
use ILC\AdminUsuarios\Models\ILCUser;
use ILC\AdminUsuarios\Providers\EventServiceProvider;
use ILC\AdminUsuarios\Providers\ILCAuthServiceProvider;
use Illuminate\Support\Facades\Auth;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Pluralizer;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Artisan;

class AdminUsuariosServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        $package
            ->name('adminusuarios')
            ->hasRoutes('api')
            ->hasConfigFile(['adminusuarios','audit', 'permission', 'sanctum', 'auth', 'excel'])
            ->hasViews()
            ->hasTranslations()
            ->hasMigrations([
                'add_user_extension_to_users_table',
                'create_audits_table',
                'create_permission_tables',
                'add_teams_fields',
                'create_personal_access_tokens_table',
                'create_user_imports_table',
                ])
            ->hasCommand(AdminUsuariosCommand::class)
            ->publishesServiceProvider(AdminUsuariosServiceProvider::class)
            ->hasInstallCommand(function(InstallCommand $command) {
                $command
                    ->startWith(function(InstallCommand $command) {
                        $command->info('instalación del paquete ILC\AdminUsuarios!');
                    })
                    ->publishConfigFile()
                    ->publishMigrations()
                   ->askToRunMigrations()
                    ->endWith(function(InstallCommand $command) {
                        $command->info('corriendo seeders...');
                        Artisan::call('db:seed', ['--class' => 'ILC\\AdminUsuarios\\Database\\Seeders\\PermissionsSeeder']);
                        $command->info(Artisan::output());
                        App::currentLocale('es');
                        Pluralizer::useLanguage('spanish');
                        $command->info('finaliza instalación!');
                    });
            });
    }

    public function bootingPackage(): void
    {
        $this->publishes([
            base_path('vendor/maatwebsite/excel/config/excel.php') => config_path('excel.php'),
        ], 'excel.php');
    }

    public function packageRegistered(): void
    {
        $this->app->singleton('auth.driver', function ($app) {
            return new ILCAuthServiceProvider($app);
        });

        $this->app->register(EventServiceProvider::class);
        $this->app->register(ILCAuthServiceProvider::class);
    }

    public function packageBooted()
    {
        // Registramos las rutas del paquete al final cuando ya se botó el paquete.
        // Esto debería asegurar que se sobreescriban las rutas del paquete base siempre
        // y cuando se llamen igual y tengan la misma firma.
        $this->app->booted(function () {
            $this->registerCustomRoutes();

            if (!$this->app->environment('production')) {
                $faker = fake();
                $faker->addProvider(new \Database\Providers\PersonaProvider($faker));
            }

        });
    }

    protected function registerCustomRoutes()
    {
        Route::middleware('api')
            ->group(__DIR__.'/../routes/api.php');
    }

}
