<?php

namespace EdgarOrozco\LaravelMailconfig;

use EdgarOrozco\LaravelMailconfig\Console\Commands\InstallMailconfig;
use EdgarOrozco\LaravelMailconfig\Providers\EventServiceProvider;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;
use Spatie\LaravelPackageTools\Exceptions\InvalidPackage;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class LaravelMailconfigServiceProvider extends PackageServiceProvider
{
    /**
     * Configura el paquete en el proveedor de servicios.
     *
     * Este método se utiliza para configurar los aspectos esenciales del paquete
     * como el nombre, si tiene vistas, traducciones, comandos y migraciones.
     */
    public function configurePackage(Package $package): void
    {
        $package
            ->name('laravel-mailconfig')
            ->hasViews()
            ->hasTranslations()
            ->hasCommand(InstallMailconfig::class)
            ->hasMigrations('create_laravel-mailconfigs_table', 'add_unaccent_extension_to_postgressql');
    }

    /**
     * Registra proveedores de servicios en la aplicación.
     *
     * Este método se utiliza para registrar servicios y vincularlos
     * en el contenedor de servicios de la aplicación Laravel.
     *
     * Aquí se registra `EventServiceProvider::class` que se necesita para
     * trabajar con los eventos en la aplicación.
     *
     * @return void
     *
     * @throws InvalidPackage
     */
    public function register()
    {
        $this->app->register(EventServiceProvider::class);

        return parent::register();
    }

    /**
     * Se invoca después de que todos los proveedores de servicios registrados se han arrancado.
     *
     * @return void
     */
    public function boot()
    {
        parent::boot();

        if (Schema::hasTable('mail_configs')) {
            LaravelMailconfig::set();
        }

        $this->registerRoutes();
        $this->bootTranslations();
        $this->validadores();
        if (env('MAILCONFIG_DB_LOG') === 'SI') {
            $this->dbLog();
        }
    }

    /**
     * Registra las rutas de la aplicación.
     *
     * Este método carga las rutas desde el archivo api.php que se encuentra
     * en el directorio de rutas del paquete.
     */
    protected function registerRoutes(): void
    {
        $this->loadRoutesFrom(__DIR__.'/routes/api.php');
    }

    /**
     * Inicializa las traducciones del paquete.
     *
     * Este método carga las traducciones desde el archivo de recursos del paquete.
     * También se configura la publicación de las traducciones para que puedan ser personalizadas.
     */
    protected function bootTranslations(): void
    {
        $this->loadTranslationsFrom(__DIR__.'/../resources/lang', 'mailconfig');

        $this->publishes([
            __DIR__.'/../resources/lang' => resource_path('lang/vendor/laravel-mailconfig'),
        ], 'mailconfig-translations');
    }

    protected function dbLog(): void
    {
        DB::listen(function ($query) {
            // Reemplazar los parámetros en la consulta SQL
            $fullQuery = vsprintf(str_replace(['%', '?'], ['%%', '%s'], $query->sql), $query->bindings);
            // Registrar la consulta completa en el log
            Log::info($fullQuery);
            Log::info('Tiempo de ejecución: '.$query->time);
        });
    }

    protected function validadores(): void
    {
        /*
         * Extiende el sistema de validación con la regla "iunique" que valida campos únicos insensibles a mayúsculas o minúsculas
        */
        Validator::extend('iunique', function ($attribute, $value, $parameters, $validator) {
            $query = DB::table($parameters[0]);
            $column = $query->getGrammar()->wrap($parameters[1]);
            if (isset($parameters[2])) {
                $query->where($parameters[2], '!=', $parameters[3]);
            }

            return ! $query->whereRaw("lower({$column}) = lower(?)", [$value])->count();
        });

    }
}
