<?php

namespace EdgarOrozco\LaravelMailconfig\Console\Commands;

use EdgarOrozco\LaravelMailconfig\Services\PermissionService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Spatie\Permission\Models\Role;

class InstallMailconfig extends Command
{
    protected $signature = 'mailconfig:install';

    protected $description = 'Install the Laravel Mailconfig package';

    protected $permissionService;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        parent::__construct();
    }

    public function handle()
    {
        $this->info('Export migrations...');
        Artisan::call('vendor:publish', ['--tag' => 'mailconfig-migrations']);

        $this->info('Running migrations...');
        Artisan::call('migrate');
        $this->info(Artisan::output());

        $this->info('Running seeders...');
        Artisan::call('db:seed', ['--class' => 'EdgarOrozco\\LaravelMailconfig\\Database\\Seeders\\PermissionsSeeder']);
        $this->info(Artisan::output());

        Artisan::call('route:clear');

        $defaultIndex = 0;
        $asociar_permisos = $this->choice(
            '¿Deseas asociar los permisos?',
            ['No', 'Si. A un rol', 'Si. A un usuario específico'],
            $defaultIndex,
            $maxAttempts = null,
            $allowMultipleSelections = false
        );
        if ($asociar_permisos === 'Si. A un rol') {
            $this->asociarPermisosaUnRol();
        }
        if ($asociar_permisos === 'Si. A un usuario específico') {
            $this->asociarPermisosaUnUsuario();
        }
        $this->info('Laravel Mailconfig installed successfully.');
    }

    public function asociarPermisosaUnRol(): void
    {
        $roles = Role::get()->pluck('name')->toArray();
        if (empty($roles)) {
            $this->error('No hay roles disponibles en el sistema.');

            return;
        }

        $selected_role = $this->choice(
            'Selecciona el rol al que deseas asociar los permisos:',
            $roles,
            $defaultIndex = 0,
            $maxAttempts = null,
            $allowMultipleSelections = false
        );

        $role = Role::where('name', $selected_role)->first();
        $permissions = $this->permissionService->getPermissions();
        foreach ($permissions as $permission) {
            $role->givePermissionTo($permission['name']);
        }
    }

    public function asociarPermisosaUnUsuario(): void
    {
        $tipo_usuario_id = $this->choice(
            '¿Proporcionar el usuario por su ID o por su email?:',
            ['Email', 'ID'],
            $defaultIndex = 0,
            $maxAttempts = null,
            $allowMultipleSelections = false
        );

        if ($tipo_usuario_id === 'Email') {
            $user_email = $this->ask('Ingresa el email del usuario:');
            $user = env('USER_MODEL')::where('email', $user_email)->first();
            if (! $user) {
                $this->error('No se encontró ningún usuario con ese email.');

                return;
            }
        }

        if ($tipo_usuario_id === 'ID') {
            $id = $this->ask('Ingresa el ID del usuario:');
            $user = env('USER_MODEL')::find($id);
            if (! $user) {
                $this->error('No se encontró ningún usuario con ese ID.');

                return;
            }
        }

        $permissions = $this->permissionService->getPermissions();
        foreach ($permissions as $permission) {
            $user->givePermissionTo($permission['name']);
        }
    }
}
