<?php

namespace EdgarOrozco\LaravelMailconfig\Models;

use EdgarOrozco\LaravelMailconfig\Traits\Filterable;
use EdgarOrozco\LaravelMailconfig\Traits\HasPackageFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class MailConfig extends Model
{
    use Filterable, HasPackageFactory;

    protected $fillable = [
        'name',
        'description',
        'driver',
        'host',
        'port',
        'username',
        'password',
        'encryption',
        'from_address',
        'from_name',
        'enabled',
    ];

    protected $casts = ['enabled' => 'boolean'];

    /**
     * Cambia el atributo 'nombre' a mayúsculas.
     *
     * @param  string  $v  Valor que se va a asignar al atributo 'nombre'.
     * @return void
     */
    public function setNameAttribute($v)
    {
        $this->attributes['name'] = mb_strtoupper($v);
    }

    /**
     * Consulta para obtener la primera configuración de correo habilitada.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder|static|null
     */
    public function scopeEnabled($query)
    {
        return $query->whereEnabled(true)->first();
    }

    /**
     * Consulta para obtener el hash de la configuración de correo habilitada.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return string|null Devuelve el hash en formato md5 del nombre de la
     *                     configuración de correo si está habilitada. Si no,
     *                     devuelve null.
     */
    public function scopeHashConfigEnabled($query)
    {
        return ($cfg = $query->whereEnabled(true)->first()) ? md5($cfg->name) : null;
    }

    protected static function boot()
    {
        parent::boot();

        static::saved(function ($model) {
            DB::table('mail_configs')
                ->where('id', $model->id)
                ->update([
                    'tsv' => DB::raw("
                        to_tsvector('spanish',
                            coalesce(name, '') || ' ' ||
                            coalesce(description, '') || ' ' ||
                            coalesce(driver, '') || ' ' ||
                            coalesce(host, '') || ' ' ||
                            coalesce(port::text, '') || ' ' ||
                            coalesce(username, '') || ' ' ||
                            coalesce(password, '') || ' ' ||
                            coalesce(encryption, '') || ' ' ||
                            coalesce(from_address, '') || ' ' ||
                            coalesce(from_name, '') || ' ' ||
                            coalesce(enabled::text, '')
                        )
                    "),
                ]);
        });
    }
}
