<?php

namespace EdgarOrozco\LaravelMailconfig\Http\Controllers;

use EdgarOrozco\LaravelMailconfig\Actions\CountEnabledConfigs;
use EdgarOrozco\LaravelMailconfig\Actions\DeleteMailConfig;
use EdgarOrozco\LaravelMailconfig\Actions\DisableExistingConfigIfEnabled;
use EdgarOrozco\LaravelMailconfig\Actions\GetAllConfigs;
use EdgarOrozco\LaravelMailconfig\Actions\GetMailConfigById;
use EdgarOrozco\LaravelMailconfig\Actions\SaveMailConfig;
use EdgarOrozco\LaravelMailconfig\Actions\ValidateMailConfigRequest;
use Illuminate\Http\Request;

class MailConfigController extends BaseController
{
    protected $getMailConfigById;

    protected $countEnabledConfigs;

    protected $getAllConfigs;

    protected $validateMailConfigRequest;

    protected $disableExistingConfigIfEnabled;

    protected $saveMailConfig;

    protected $deleteMailConfig;

    public function __construct(
        CountEnabledConfigs $countEnabledConfigs,
        GetAllConfigs $getAllConfigs,
        ValidateMailConfigRequest $validateMailConfigRequest,
        DisableExistingConfigIfEnabled $disableExistingConfigIfEnabled,
        SaveMailConfig $saveMailConfig,
        GetMailConfigById $getMailConfigById,
        DeleteMailConfig $deleteMailConfig
    ) {
        $this->getMailConfigById = $getMailConfigById;
        $this->countEnabledConfigs = $countEnabledConfigs;
        $this->getAllConfigs = $getAllConfigs;
        $this->validateMailConfigRequest = $validateMailConfigRequest;
        $this->disableExistingConfigIfEnabled = $disableExistingConfigIfEnabled;
        $this->saveMailConfig = $saveMailConfig;
        $this->deleteMailConfig = $deleteMailConfig;
    }

    /**
     * Form to create a new configuration and lists configurations stored in DB
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View|\Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $config = null;

        // Si se recibe el parametro id, entonces se busca el objeto crorespondiente y se devuelve
        if ($request->get('id')) {
            $data = ($this->getMailConfigById)($request->get('id'));

            return $this->respond($request, compact('data'), 'mailconfig::index');
        }

        // Se cuenta cuántos registros de configuración están habilitados
        $is_enabled = ($this->countEnabledConfigs)();

        // Se filtran los registros
        $query = ($this->getAllConfigs)($request);

        // Si viene una búsqueda por el filtro de búsqueda general ejecutamos el fulltext search
        if ($search = $request->input('search')) {
            // Si viene más de una palabra en la cadena search entonces separamos las palabras y agregamos :* para
            // crear una búsqueda parcial para cada palabra. ademas de separar las palabras por &
            $search = explode(' ', $search);
            $search = array_map(function ($word) {
                return $word.':*';
            }, $search);
            $search = implode(' & ', $search);
            $query->whereRaw("tsv @@ to_tsquery('spanish', ?)", [$search]);
        }

        // Finalmente se paginan y se devuelven los resultados
        $data = $query->paginate($request->input('per_page', 10));

        // Se genera la respuesta.
        return $this->respond($request, compact('data', 'config', 'is_enabled'), 'mailconfig::index');
    }

    /**
     * Create or update mail configuration in DB
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        ($this->validateMailConfigRequest)($request);

        $data = $request->all();
        ($this->disableExistingConfigIfEnabled)($data['enabled']);
        ($this->saveMailConfig)($request->get('id'), $data);

        return $this->respond($request, [
            'success' => true,
            'message' => __('mailconfig::saved-ok'),
        ], '', true);
    }

    /**
     * Destroy a disabled configuration
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request)
    {
        $cfg = ($this->getMailConfigById)($request->get('id'));

        ($this->deleteMailConfig)($cfg);

        return $this->respond($request, [
            'success' => true,
            'message' => __('mailconfig::deleted-ok'),
        ], '', true);
    }
}
