import {defineStore} from 'pinia';
import {ref} from 'vue';
import apiClient from '@/plugins/axios';
import {useLocalStorage} from '@vueuse/core';

export const useAuthStore = defineStore('auth', () => {
  // Estado
  const user = ref(useLocalStorage('user', null));
  const isAuthenticated = ref(useLocalStorage('isAuthenticated', false));

  // Acciones
  async function login(credentials) {
    await apiClient.get('/sanctum/csrf-cookie'); // Obteniendo la cookie de CSRF
    await apiClient.post('/api/login', credentials);
    await fetchUser();
  }

  async function fetchUser() {
    try {
      const response = await apiClient.get('/api/user');
      user.value = response.data;
      isAuthenticated.value = true;
    } catch (error) {
      user.value = null;
      isAuthenticated.value = false;
      console.error('Failed to fetch user:', error);
    }
  }

  async function logout() {
    try {
      await apiClient.post('/api/logout');
      user.value = null;
      isAuthenticated.value = false;
    } catch (error) {
      console.error('Logout failed:', error);
    }
  }

  return {user, isAuthenticated, login, fetchUser, logout};
}, {
  persist: {
    key: 'auth',

    // Con localStorage la sesión es persistente en el storage del navegador. No se pierden los datos al cerrar el navegador
    storage: window.localStorage,

    // Con sessionStorage La sesión está asociada a la sesión en el navegador. Si se cierra el navegador los datos de sesión se pierden.
    // storage: window.sessionStorage,
  }
});
