// router/index.js
import { createRouter, createWebHistory } from 'vue-router';
import { useAuthStore } from '@/store/auth';

// Usamos import.meta.glob para cargar automáticamente todas las rutas en el directorio `router`.
const routeModules = import.meta.glob('@app/router/*.js', { eager: true });

const dynamicRoutes = [];

// Iteramos sobre los módulos de rutas importados y los combinamos en un solo array
for (const path in routeModules) {
  if (path !== 'index.js') {
    const routeConfig = routeModules[path].default;
    dynamicRoutes.push(...routeConfig);
  }
}

// Definimos las rutas básicas comunes
const baseRoutes = [
  {
    path: '/',
    name: 'Home',
    meta: {
      layout: 'content',
      requiresAuth: true,
    },
  },
  {
    path: '/login',
    name: 'Login',
    component: () => import('@/views/Login.vue'),
    meta: {
      layout: 'blank',
      requiresAuth: false,
    },
  },
  {
    path: '/:pathMatch(.*)*',
    name: 'NotFound',
    component: () => import('@/views/NotFound.vue'),
    meta: {
      layout: 'blank',
    },
  },
];

// Combinamos las rutas base con las dinámicas
const routes = [...baseRoutes, ...dynamicRoutes];

const router = createRouter({
  history: createWebHistory('/'),
  scrollBehavior: () => {
    // always scroll to top
    return { top: 0, behavior: 'smooth' };
  },
  routes,
});

router.beforeEach((to, from, next) => {
  const authStore = useAuthStore();

  // Redirigimos a home si el usuario intenta acceder a /login y ya está autenticado
  if (to.name === 'Login' && authStore.isAuthenticated) {
    next({ name: 'Home' });
  } else if (to.meta.requiresAuth && !authStore.isAuthenticated) {
    // Si la ruta requiere autenticación y el usuario no está autenticado
    next({ name: 'Login' }); // Redirigir al login
  } else {
    next(); // Continuar con la navegación
  }
});

router.afterEach((to, from) => {
  const toDepth = to.path.split('/').length;
  const fromDepth = from.path.split('/').length;
  to.meta.animationIn = toDepth < fromDepth ? 'animate__fadeInLeft' : 'animate__fadeInRight';
  to.meta.animationOut = toDepth < fromDepth ? 'animate__fadeOutRight' : 'animate__fadeOutLeft';
});

export default router;
